<?php

namespace Limb_Chatbot\Includes\Utilities;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Interfaces\Utility_Interface;

/**
 * Global Slack Channels Utility
 *
 * Base utility for managing Slack channel operations.
 * Acts as a context holder for channel-related operations.
 *
 * @package Limb_Chatbot\Includes\Utilities
 * @since 1.0.0
 */
class Slack_Channels_Utility extends Utility implements Utility_Interface {

	/**
	 * Utility name.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public static string $name = 'Slack Channels';

	/**
	 * Slack configuration object.
	 *
	 * @var Config|null
	 * @since 1.0.0
	 */
	public ?Config $config = null;

	/**
	 * Slack integration ID.
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $integration_id = 'slack';

	/**
	 * Channel types to fetch.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $types = 'public_channel,private_channel';

	/**
	 * Whether to exclude archived channels.
	 *
	 * @var bool
	 * @since 1.0.0
	 */
	public bool $exclude_archived = true;

	/**
	 * Specific channel ID for single channel operations.
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $channel_id = null;

	/**
	 * HTTP request timeout in seconds.
	 *
	 * @var int
	 * @since 1.0.0
	 */
	public int $timeout = 30;

	/**
	 * Constructor.
	 *
	 * @param Config|null $config Slack configuration object.
	 * @param string|null $integration_id Integration identifier.
	 *
	 * @since 1.0.0
	 */
	public function __construct( ?Config $config = null, ?string $integration_id = null ) {
		$this->config         = $config;
		$this->integration_id = $integration_id ?? 'slack';
	}

	/**
	 * Get integration ID.
	 *
	 * @return string|null
	 * @since 1.0.0
	 */
	public function get_integration_id(): ?string {
		return $this->integration_id;
	}

	/**
	 * Get configuration object.
	 *
	 * @return Config|null
	 * @since 1.0.0
	 */
	public function get_config(): ?Config {
		return $this->config;
	}

	/**
	 * Get timeout value.
	 *
	 * @return int
	 * @since 1.0.0
	 */
	public function get_timeout(): int {
		return $this->timeout;
	}
}

