<?php

namespace Limb_Chatbot\Includes\Utilities;

use Limb_Chatbot\Includes\Data_Objects\AI_Model;
use Limb_Chatbot\Includes\Data_Objects\AI_Model_Meta;
use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Interfaces\Utility_Interface;


/**
 * Class Fine_Tuning_Utility
 *
 * Utility class for managing fine-tuning operations related to AI models.
 *
 * @package Limb_Chatbot\Includes\Utilities
 *
 * @since 1.0.0
 */
class Fine_Tuning_Utility extends Utility implements Utility_Interface {

	/**
	 * The AI model associated with this utility.
	 *
	 * @var AI_Model
	 * @since 1.0.0
	 */
	public AI_Model $model;

	/**
	 * Optional configuration related to the AI model.
	 *
	 * @var Config|null
	 * @since 1.0.0
	 */
	public ?Config $config = null;

	/**
	 * AI provider identifier.
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $ai_provider_id;

	/**
	 * Static utility name.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public static string $name = 'Fine Tuning Utility';

	/**
	 * Constructor.
	 *
	 * Initializes the Fine_Tuning_Utility with optional model, config, and AI provider ID.
	 *
	 * @param AI_Model|null $model AI model instance. If null, must be set later.
	 * @param Config|null $config Optional configuration object.
	 * @param string|null $ai_provider_id Optional AI provider ID; if null, derived from the model.
	 *
	 * @since 1.0.0
	 */
	public function __construct( Ai_Model $model = null, ?Config $config = null, ?string $ai_provider_id = null ) {
		$this->model          = $model;
		$this->ai_provider_id = $ai_provider_id;
		if ( is_null( $this->ai_provider_id ) ) {
			$this->ai_provider_id = $this->model->get_ai_provider_id();
		}
		if ( ! is_null( $config ) ) {
			$this->config = $config;
		}
		if ( is_null( $this->config ) ) {
			if ( $config_meta = AI_Model_Meta::where( [ 'ai_model_id' => $this->model->get_id(), 'meta_key' => 'config_id' ] )->first() ) {
				$this->config = Config::find( $config_meta->meta_value );
			}
		}
	}

	/**
	 * Get the AI provider ID associated with this utility.
	 *
	 * @return string|null AI provider identifier or null if not set.
	 *
	 * @since 1.0.0
	 */
	public function get_ai_provider_id(): ?string {
		return $this->ai_provider_id;
	}

	/**
	 * Get the configuration object.
	 *
	 * @return Config|null Config object or null if not set.
	 *
	 * @since 1.0.0
	 */
	public function get_config(): ?Config {
		return $this->config;
	}
}