<?php

namespace Limb_Chatbot\Includes\Utilities;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Data_Objects\File;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Utility_Interface;

/**
 * File Utility for AI Provider File Operations
 *
 * Base utility class for handling file-related operations with AI providers.
 * This class provides the foundation for provider-specific file utilities
 * and handles common file operations like upload and refresh.
 *
 * This utility is used by:
 * - File_Service for uploading files to AI providers
 * - Background processes for file management
 * - Provider-specific utilities (OpenAI, Gemini, etc.)
 *
 * @package Limb_Chatbot\Includes\Utilities
 * @since 1.0.0
 */
class File_Utility extends Utility implements Utility_Interface {

	/**
	 * Purpose of the file utility instance.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $purpose;

	/**
	 * Path to the file.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $file_path;

	/**
	 * AI provider identifier.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $ai_provider_id;

	/**
	 * Optional configuration object.
	 *
	 * @var Config|null
	 * @since 1.0.0
	 */
	public ?Config $config;

	/**
	 * Static utility name.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public static string $name = 'File Utility';

	/**
	 * Constructor.
	 *
	 * Initializes the File_Utility instance.
	 *
	 * @param  string|null  $purpose  Purpose for which this utility is used.
	 * @param  string|null  $file_path  Path of the file to operate on.
	 * @param  string|null  $ai_provider_id  AI provider identifier associated with this file.
	 * @param  Config|null  $config  Optional configuration object.
	 *
	 * @since 1.0.0
	 */
	public function __construct( ?string $purpose, ?string $file_path, ?string $ai_provider_id, ?Config $config = null ) {
		$this->purpose        = $purpose;
		$this->file_path      = $file_path;
		$this->ai_provider_id = $ai_provider_id;
		$this->config         = $config;
	}

	/**
	 * Get the purpose of this file utility.
	 *
	 * @return string The purpose string.
	 *
	 * @since 1.0.0
	 */
	public function get_purpose(): string {
		return $this->purpose;
	}

	/**
	 * Set the purpose of this file utility.
	 *
	 * @param string $purpose The purpose string to set.
	 *
	 * @since 1.0.0
	 */
	public function set_purpose( string $purpose ): void {
		$this->purpose = $purpose;
	}

	/**
	 * Get the file path.
	 *
	 * @return string The file path string.
	 *
	 * @since 1.0.0
	 */
	public function get_file_path(): string {
		return $this->file_path;
	}

	/**
	 * Set the file path.
	 *
	 * @param string $file_path The file path string to set.
	 *
	 * @since 1.0.0
	 */
	public function set_file_path( string $file_path ): void {
		$this->file_path = $file_path;
	}

	/**
	 * Get the AI provider ID.
	 *
	 * @return string AI provider identifier.
	 *
	 * @since 1.0.0
	 */
	public function get_ai_provider_id(): string {
		return $this->ai_provider_id;
	}

	/**
	 * Get the optional config object.
	 *
	 * @return Config|null The config instance or null if not set.
	 *
	 * @since 1.0.0
	 */
	public function get_config(): ?Config {
		return $this->config;
	}

	/**
	 * Uploads a file to the AI provider.
	 *
	 * Delegates the upload operation to the provider-specific utility.
	 * The actual upload implementation depends on the AI provider being used.
	 *
	 * @param File $file The File object to upload.
	 * @return array Array of Meta_Data_Object instances with provider-specific metadata.
	 * @throws Exception If upload fails or provider doesn't support file uploads.
	 * @since 1.0.0
	 */
	public function upload( File $file ) {
		return $this->get_ai_provider_utility()->upload( $file );
	}

	/**
	 * Refreshes file metadata from the AI provider.
	 *
	 * Retrieves the latest metadata for a file from the AI provider
	 * and returns updated metadata objects.
	 *
	 * @param File $file The File object to refresh.
	 * @return array Array of Meta_Data_Object instances with updated metadata.
	 * @throws Exception If refresh fails or provider doesn't support file operations.
	 * @since 1.0.0
	 */
	public function refresh( File $file ) {
		return $this->get_ai_provider_utility()->refresh( $file );
	}
}