<?php

namespace Limb_Chatbot\Includes\Utilities;

use Limb_Chatbot\Includes\Data_Objects\Vector;
use Limb_Chatbot\Includes\Interfaces\Utility_Interface;


/**
 * Class Embedding_Utility
 *
 * Utility class for handling vector embeddings with optional AI provider context.
 *
 * @package Limb_Chatbot\Includes\Utilities
 *
 * @property Vector $vector The vector instance this utility operates on.
 * @property string|null $ai_provider_id The AI provider identifier associated with this vector.
 */
class Embedding_Utility extends Utility implements Utility_Interface {

	/**
	 * The vector instance.
	 *
	 * @var Vector
	 */
	public Vector $vector;

	/**
	 * AI provider identifier string.
	 *
	 * @var string|null
	 */
	public ?string $ai_provider_id;

	/**
	 * Static utility name.
	 *
	 * @var string
	 */
	public static string $name = 'Embedding Utility';

	/**
	 * Constructor.
	 *
	 * Initializes the utility with a Vector instance.
	 * Automatically sets the AI provider ID from the vector's AI model.
	 *
	 * @param Vector $vector The vector object to be used by this utility.
	 */
	public function __construct( Vector $vector ) {
		$this->set_vector( $vector );
		$this->set_ai_provider_id( $this->vector->get_ai_model()->get_ai_provider_id() );
	}

	/**
	 * Get the vector instance.
	 *
	 * @return Vector The current vector object.
	 */
	public function get_vector(): Vector {
		return $this->vector;
	}

	/**
	 * Set the vector instance.
	 *
	 * @param Vector $vector The new vector object to set.
	 */
	public function set_vector( Vector $vector ): void {
		$this->vector = $vector;
	}

	/**
	 * Set the AI provider ID.
	 *
	 * @param string|null $ai_provider_id The AI provider identifier to set.
	 */
	public function set_ai_provider_id( ?string $ai_provider_id ): void {
		$this->ai_provider_id = $ai_provider_id;
	}

	/**
	 * Get the AI provider ID.
	 *
	 * @return string|null The AI provider identifier, or null if none is set.
	 */
	public function get_ai_provider_id(): ?string {
		return $this->ai_provider_id;
	}
}