<?php

namespace Limb_Chatbot\Includes\Services\Notifications\Types;

use Limb_Chatbot\Includes\Data_Objects\Chatbot_User;
use Limb_Chatbot\Includes\Interfaces\Notification_Type_Interface;
use Limb_Chatbot\Includes\Services\Notifications\Notification_Envelope;

/**
 * Base implementation for notification types.
 *
 * Subclasses define key, target roles, channels, and build_envelope() for dynamic content.
 *
 * @package Limb_Chatbot\Includes\Services\Notifications\Types
 * @since 1.0.15
 */
abstract class Abstract_Notification_Type implements Notification_Type_Interface {

	/**
	 * Default channels for this type (override in subclass if needed).
	 *
	 * @var string[]
	 * @since 1.0.15
	 */
	protected array $channels = [ 'email' ];

	/**
	 * Default target roles (override in subclass).
	 *
	 * @return string[]
	 * @since 1.0.15
	 */
	abstract public function get_target_roles(): array;

	/**
	 * @inheritDoc
	 */
	public function get_channels(): array {
		return $this->channels;
	}

	/**
	 * Default: queued (not instant). Override in subclass for instant notifications.
	 *
	 * @return bool
	 * @since 1.0.15
	 */
	public function is_instant(): bool {
		return false;
	}

	/**
	 * Create an envelope instance (helper for subclasses).
	 *
	 * @param  string  $channel_key  Channel key.
	 * @param  string  $subject  Subject.
	 * @param  string  $body  Body.
	 * @param  array  $payload  Optional payload.
	 * @param  string  $content_type  Content type (e.g. 'text/plain', 'text/html').
	 *
	 * @return Notification_Envelope
	 * @since 1.0.15
	 */
	protected function create_envelope( string $channel_key, string $subject, string $body, array $payload = [], string $content_type = 'text/plain' ): Notification_Envelope {
		return new Notification_Envelope(
			$this->get_key(),
			$channel_key,
			$subject,
			$body,
			$payload,
			$content_type
		);
	}
}
