<?php

namespace Limb_Chatbot\Includes\Services\Notifications;

/**
 * Value object holding prepared notification content and metadata for a single channel.
 *
 * Built by a Notification_Type from context payload; consumed by a Notification_Channel.
 *
 * @package Limb_Chatbot\Includes\Services\Notifications
 * @since 1.0.15
 */
class Notification_Envelope {

	/**
	 * Notification type key (e.g. 'chat_created').
	 *
	 * @var string
	 * @since 1.0.15
	 */
	private string $type_key;

	/**
	 * Channel key (e.g. 'email').
	 *
	 * @var string
	 * @since 1.0.15
	 */
	private string $channel_key;

	/**
	 * Subject line (e.g. email subject).
	 *
	 * @var string
	 * @since 1.0.15
	 */
	private string $subject;

	/**
	 * Body content (e.g. email body, plain or HTML).
	 *
	 * @var string
	 * @since 1.0.15
	 */
	private string $body;

	/**
	 * Optional structured payload for the channel (e.g. headers, attachments).
	 *
	 * @var array
	 * @since 1.0.15
	 */
	private array $payload;

	/**
	 * Content type of the body (e.g. 'text/plain', 'text/html').
	 *
	 * @var string
	 * @since 1.0.15
	 */
	private string $content_type;

	/**
	 * Constructor.
	 *
	 * @param  string  $type_key  Notification type key.
	 * @param  string  $channel_key  Channel key.
	 * @param  string  $subject  Subject.
	 * @param  string  $body  Body.
	 * @param  array  $payload  Optional payload.
	 * @param  string  $content_type  Content type (e.g. 'text/plain', 'text/html'). Default 'text/plain'.
	 *
	 * @since 1.0.15
	 */
	public function __construct( string $type_key, string $channel_key, string $subject, string $body, array $payload = [], string $content_type = 'text/plain' ) {
		$this->type_key     = $type_key;
		$this->channel_key  = $channel_key;
		$this->subject      = $subject;
		$this->body         = $body;
		$this->payload      = $payload;
		$this->content_type = $content_type;
	}

	/**
	 * Get notification type key.
	 *
	 * @return string
	 * @since 1.0.15
	 */
	public function get_type_key(): string {
		return $this->type_key;
	}

	/**
	 * Get channel key.
	 *
	 * @return string
	 * @since 1.0.15
	 */
	public function get_channel_key(): string {
		return $this->channel_key;
	}

	/**
	 * Get subject.
	 *
	 * @return string
	 * @since 1.0.15
	 */
	public function get_subject(): string {
		return $this->subject;
	}

	/**
	 * Get body.
	 *
	 * @return string
	 * @since 1.0.15
	 */
	public function get_body(): string {
		return $this->body;
	}

	/**
	 * Get payload.
	 *
	 * @return array
	 * @since 1.0.15
	 */
	public function get_payload(): array {
		return $this->payload;
	}

	/**
	 * Get content type (e.g. 'text/plain', 'text/html').
	 *
	 * @return string
	 * @since 1.0.15
	 */
	public function get_content_type(): string {
		return $this->content_type;
	}
}
