<?php

namespace Limb_Chatbot\Includes\Services\Notifications;

use Limb_Chatbot\Includes\Interfaces\Notification_Channel_Interface;

/**
 * Registry of notification channels (email, push, etc.).
 *
 * Channels are registered in code; dispatch uses the appropriate channel by key.
 *
 * @package Limb_Chatbot\Includes\Services\Notifications
 * @since 1.0.15
 */
class Notification_Channel_Registry {

	/**
	 * Registered channels by key.
	 *
	 * @var array<string, Notification_Channel_Interface>
	 * @since 1.0.15
	 */
	private array $channels = [];

	/**
	 * Register a channel.
	 *
	 * @param  Notification_Channel_Interface  $channel  Channel instance.
	 *
	 * @return void
	 * @since 1.0.15
	 */
	public function register( Notification_Channel_Interface $channel ): void {
		$this->channels[ $channel->get_channel_key() ] = $channel;
	}

	/**
	 * Get a channel by key.
	 *
	 * @param  string  $key  Channel key.
	 *
	 * @return Notification_Channel_Interface|null
	 * @since 1.0.15
	 */
	public function get( string $key ): ?Notification_Channel_Interface {
		return $this->channels[ $key ] ?? null;
	}

	/**
	 * Get all registered channels.
	 *
	 * @return Notification_Channel_Interface[]
	 * @since 1.0.15
	 */
	public function get_all(): array {
		return array_values( $this->channels );
	}

	/**
	 * Check if a channel is registered.
	 *
	 * @param  string  $key  Channel key.
	 *
	 * @return bool
	 * @since 1.0.15
	 */
	public function has( string $key ): bool {
		return isset( $this->channels[ $key ] );
	}
}
