<?php

namespace Limb_Chatbot\Includes\Services\Knowledge\Source_Validators;

use Limb_Chatbot\Includes\Data_Objects\File;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Source_Validator_Interface;

class File_Validator implements Source_Validator_Interface {

	/**
	 * Validates the parameters for file-based datasets.
	 *
	 * Ensures that files are selected and the chosen AI model supports file inputs.
	 *
	 * @param  array $config Configuration array containing file dataset parameters.
	 *
	 * @throws Exception If files are missing, invalid, or if the AI model is incompatible.
	 *
	 * @since 1.0.0
	 */
	public function validate( array $config ): void {
		$files = $config['files'] ?? null;
		if ( empty( $files ) || ! is_array( $files ) ) {
			throw new Exception( Error_Codes::VALIDATION_INVALID_VALUE, __( 'Files are missing.', 'limb-chatbot' ) );
		}
		foreach ( $files as $file_uuid ) {
			$file = File::find_by_uuid( $file_uuid );
			if ( ! $file instanceof File ) {
				throw new Exception( Error_Codes::VALIDATION_INVALID_VALUE, __( 'One or more selected files are invalid. Please verify your file selection.', 'limb-chatbot' ) );
			}
		}
	}

	/**
	 * Gets the total number of files to be included in the dataset.
	 *
	 * @param  array $config Configuration array containing file dataset parameters.
	 *
	 * @return int Total number of files.
	 *
	 * @since 1.0.9
	 */
	public function get_total( array $config ): int {
		return count( $config['files'] ?? [] );
	}
}