<?php

namespace Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers;

use Limb_Chatbot\Includes\Data_Objects\Dataset;
use Limb_Chatbot\Includes\Interfaces\Source_Fetcher_Interface;

/**
 * URL Source Fetcher for dataset generation.
 *
 * Fetches URL sources for dataset generation.
 * Returns multiple items, one for each URL in the array.
 *
 * @package Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers
 * @since 1.0.0
 */
class URL_Source_Fetcher implements Source_Fetcher_Interface {

	/**
	 * Fetch dataset entries from URL sources.
	 *
	 * Returns multiple items, one for each URL in the urls array.
	 * Supports pagination through the $item parameter.
	 *
	 * @param  array  $item  Arguments that control fetching behavior (paged, per_page).
	 * @param  array  $config  Configuration array containing urls array.
	 *
	 * @return array[] Structured dataset entries.
	 * @since 1.0.0
	 */
	public function fetch( array $item, array $config ): array {
		$urls = $config['urls'] ?? null;

		if ( empty( $urls ) || ! is_array( $urls ) ) {
			return [];
		}

		// Support pagination
		$paged = $item['paged'] ?? 1;
		$per_page = $item['per_page'] ?? 5;
		$offset = ( $paged - 1 ) * $per_page;
		$urls_slice = array_slice( $urls, $offset, $per_page );

		$result = [];
		foreach ( $urls_slice as $url ) {
			if ( ! empty( $url ) && is_string( $url ) ) {
				$result[] = [
					'source'          => $url,
					'source_type'     => Dataset::SOURCE_URL,
					'source_sub_type' => null,
				];
			}
		}

		return $result;
	}
}

