<?php

namespace Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers;

use Limb_Chatbot\Includes\Interfaces\Knowledge_Mapper_Interface;

/**
 * Knowledge mapper for URL sources.
 *
 * Extracts structured input (title) and output (content) from a URL source
 * for dataset entry generation.
 *
 * @since 1.0.0
 */
class URL_Knowledge_Mapper implements Knowledge_Mapper_Interface {

	/**
	 * Generate the dataset entry input from a URL source.
	 *
	 * Returns the URL or a title extracted from the URL.
	 *
	 * @param  mixed  $object  The source object, expected to be a URL string or array with URL data.
	 *
	 * @return string         The dataset input, or an empty string if invalid.
	 *
	 * @since 1.0.0
	 */
	public function get_entry_input( $object ): string {
		if ( empty( $object ) ) {
			return '';
		}

		// If it's a string (URL), return it
		if ( is_string( $object ) ) {
			return $object;
		}

		// If it's an array with title, return title
		if ( is_array( $object ) && isset( $object['title'] ) ) {
			return trim( $object['title'] );
		}

		// If it's an array with url, return url
		if ( is_array( $object ) && isset( $object['url'] ) ) {
			return trim( $object['url'] );
		}

		return '';
	}

	/**
	 * Generate the dataset output content from a URL source.
	 *
	 * Returns the content extracted from the URL.
	 *
	 * @param  mixed  $object  The source object, expected to be an array with 'content' key.
	 *
	 * @return string         The formatted output content, or an empty string if invalid.
	 *
	 * @since 1.0.0
	 */
	public function get_entry_output( $object ): string {
		if ( empty( $object ) || ! is_array( $object ) ) {
			return '';
		}

		if ( isset( $object['content'] ) ) {
			return wp_kses_post( $object['content'] );
		}

		return '';
	}
}

