<?php

namespace Limb_Chatbot\Includes\Services\Chatbot_Analytics\Types;

/**
 * Class Action_Analytics_Type
 *
 * Represents an analytics type based on a chatbot action.
 * Dynamically created from action objects to track action-specific metrics.
 *
 * @package Limb_Chatbot\Includes\Services\Chatbot_Analytics\Types
 * @since 1.0.0
 */
class Action_Analytics_Type implements Analytics_Type {

	/**
	 * The action ID (unique identifier for this analytics type).
	 *
	 * @var string
	 * @since 1.0.0
	 */
	private string $action_id;

	/**
	 * The action title (human-readable label).
	 *
	 * @var string
	 * @since 1.0.0
	 */
	private string $action_title;

	/**
	 * The action name (machine-readable name).
	 *
	 * @var string
	 * @since 1.0.0
	 */
	private string $action_name;

	/**
	 * Action_Analytics_Type constructor.
	 *
	 * @param string    $action_id    The action ID
	 * @param string $action_title The action title (human-readable)
	 * @param string $action_name  The action name (machine-readable)
	 *
	 * @since 1.0.0
	 */
	public function __construct( string $action_id, string $action_title, string $action_name ) {
		$this->action_id    = $action_id;
		$this->action_title = $action_title;
		$this->action_name  = $action_name;
	}

	/**
	 * Get the unique identifier for this analytics type (the action ID).
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_id(): string {
		return (string) $this->action_id;
	}

	/**
	 * Get the human-readable label for this analytics type (action title).
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_label(): string {
		return $this->action_title;
	}

	/**
	 * Get the machine-readable name for this analytics type (action name).
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_name(): string {
		return $this->action_name;
	}

	/**
	 * Get the group/category this analytics type belongs to (always 'actions').
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_group(): string {
		return 'actions';
	}

	/**
	 * Determine if this is a built-in type (always false for action types).
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function is_builtin(): bool {
		return false;
	}
}
