<?php

namespace Limb_Chatbot\Includes\Services\Chatbot_Analytics\Data;

/**
 * Class Analytics_Data_Point
 *
 * Represents a single data point in an analytics timeline.
 * Immutable value object ensuring data integrity.
 *
 * @package Limb_Chatbot\Includes\Services\Chatbot_Analytics\Data
 * @since 1.0.0
 */
class Analytics_Data_Point {

	/**
	 * Time label (e.g., '12 AM', '4 AM', '8 AM', etc.).
	 *
	 * @var string
	 * @since 1.0.0
	 */
	private string $time;

	/**
	 * The metric value at this time point.
	 *
	 * @var int|float
	 * @since 1.0.0
	 */
	private $value;

	/**
	 * Analytics_Data_Point constructor.
	 *
	 * @param string   $time  Time label
	 * @param int|float $value The metric value
	 *
	 * @since 1.0.0
	 */
	public function __construct( string $time, $value ) {
		$this->time  = $time;
		$this->value = is_numeric( $value ) ? $value : 0;
	}

	/**
	 * Get the time label.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_time(): string {
		return $this->time;
	}

	/**
	 * Get the metric value.
	 *
	 * @return int|float
	 * @since 1.0.0
	 */
	public function get_value() {
		return $this->value;
	}

	/**
	 * Convert to array representation.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function to_array(): array {
		return [
			'time'  => $this->time,
			'value' => $this->value,
		];
	}

	/**
	 * Create from array.
	 *
	 * @param array $data Array with 'time' and 'value' keys
	 *
	 * @return self
	 * @since 1.0.0
	 */
	public static function from_array( array $data ): self {
		return new self(
			$data['time'] ?? '',
			$data['value'] ?? 0
		);
	}
}
