<?php

namespace Limb_Chatbot\Includes\Services\Actions\Parameter_Types;

use Limb_Chatbot\Includes\Data_Objects\Parameter;

/**
 * Class Textarea_Parameter_Type
 *
 * Handles textarea parameter validation and sanitization.
 * Similar to text type but preserves newlines and whitespace formatting.
 * Supports validation rules: min_length, max_length.
 *
 * @package Limb_Chatbot\Includes\Services\Actions\Parameter_Types
 * @since 1.0.0
 */
class Textarea_Parameter_Type extends Abstract_Parameter_Type {

	/**
	 * @inheritDoc
	 */
	public function get_type(): string {
		return Parameter::TYPE_TEXTAREA;
	}

	/**
	 * @inheritDoc
	 */
	public function sanitize( string $value, Parameter $parameter ): string {
		// Remove HTML/PHP tags but preserve newlines and formatting
		return wp_kses_post( trim( $value ) );
	}

	/**
	 * @inheritDoc
	 */
	public function validate( string $value, Parameter $parameter ): bool {
		$this->clear_errors();

		$trimmed_value = trim( $value );

		// Check if field is empty
		if ( empty( $trimmed_value ) ) {
			// If empty and required, show required error
			if ( $this->is_required( $parameter ) ) {
				$this->add_error( sprintf(
					__( '%s is required.', 'limb-chatbot' ),
					$this->get_label( $parameter )
				) );
				return false;
			}
			// If empty and not required, it's valid
			return true;
		}

		// Field has value, validate length rules
		$rules = $parameter->get_validation_rules();
		if ( ! empty( $rules ) ) {
			// Validate min length
			if ( isset( $rules['min_length'] ) && ! $this->validate_min_length( $trimmed_value, (int) $rules['min_length'], $parameter ) ) {
				return false;
			}

			// Validate max length
			if ( isset( $rules['max_length'] ) && ! $this->validate_max_length( $trimmed_value, (int) $rules['max_length'], $parameter ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Validate minimum length
	 *
	 * @param  string  $value  Value to validate
	 * @param  int  $min_length  Minimum length
	 * @param  Parameter  $parameter  Parameter configuration
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	private function validate_min_length( string $value, int $min_length, Parameter $parameter ): bool {
		if ( mb_strlen( $value ) < $min_length ) {
			$this->add_error( sprintf(
				__( '%s must be at least %d characters long.', 'limb-chatbot' ),
				$this->get_label( $parameter ),
				$min_length
			) );

			return false;
		}

		return true;
	}

	/**
	 * Validate maximum length
	 *
	 * @param  string  $value  Value to validate
	 * @param  int  $max_length  Maximum length
	 * @param  Parameter  $parameter  Parameter configuration
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	private function validate_max_length( string $value, int $max_length, Parameter $parameter ): bool {
		if ( mb_strlen( $value ) > $max_length ) {
			$this->add_error( sprintf(
				__( '%s must not exceed %d characters.', 'limb-chatbot' ),
				$this->get_label( $parameter ),
				$max_length
			) );

			return false;
		}

		return true;
	}

	/**
	 * @inheritDoc
	 */
	public function get_hint( Parameter $parameter ): ?string {
		$placeholder = $parameter->get_placeholder();

		if ( empty( $placeholder ) ) {
			return null;
		}

		return sprintf(
			__( 'For example: %s', 'limb-chatbot' ),
			$placeholder
		);
	}

	public function get_config_schema(): array {
		return array(
			'lead_capture_map_field' => array(
				'type'        => 'integer',
				'description' => 'Lead capture map field',
				'required'    => false,
			),
		);
	}
}

