<?php

namespace Limb_Chatbot\Includes\Services\Actions\Parameter_Types;

use Limb_Chatbot\Includes\Data_Objects\Parameter;

/**
 * Class Buttons_Parameter_Type
 *
 * Handles buttons/choice parameter validation and sanitization.
 * Allows users to select from predefined options stored in the config field.
 * Each option can have a label, emoji, and optional tooltip.
 *
 * Config structure example:
 * {
 *   "options": [
 *     {
 *       "value": "vegetarian",
 *       "label": "🥗 Vegetarian",
 *       "tooltip": "Plant-based options"
 *     },
 *     {
 *       "value": "vegan",
 *       "label": "🌱 Vegan",
 *       "tooltip": "100% plant-based"
 *     }
 *   ]
 * }
 *
 * @package Limb_Chatbot\Includes\Services\Actions\Parameter_Types
 * @since 1.0.0
 */
class Buttons_Parameter_Type extends Abstract_Parameter_Type {

	/**
	 * @inheritDoc
	 */
	public function get_type(): string {
		return Parameter::TYPE_BUTTONS;
	}

	/**
	 * @inheritDoc
	 */
	public function sanitize( string $value, Parameter $parameter ): string {
		$value = trim( $value );

		// Try to find matching option by value or label
		$config = $parameter->get_config();
		if ( ! empty( $config['options'] ) && is_array( $config['options'] ) ) {
			foreach ( $config['options'] as $option ) {
				// Match by value (exact, case-insensitive)
				if ( ! empty( $option['value'] ) && strtolower( $option['value'] ) === strtolower( $value ) ) {
					return $option['value'];
				}

				// Match by label (exact match for emoji or case-insensitive for text)
				if ( ! empty( $option['label'] ) ) {
					// First try exact match (handles emoji perfectly)
					if ( $option['label'] === $value ) {
						return $option['value'];
					}
					
					// Then try case-insensitive match (for text labels)
					if ( strtolower( $option['label'] ) === strtolower( $value ) ) {
						return $option['value'];
					}
				}
			}
		}

		// Return original value if no match found (validation will handle it)
		return $value;
	}

	/**
	 * @inheritDoc
	 */
	public function validate( string $value, Parameter $parameter ): bool {
		$this->clear_errors();

		$trimmed_value = trim( $value );

		// Check if field is empty
		if ( empty( $trimmed_value ) ) {
			// If empty and required, show required error
			if ( $this->is_required( $parameter ) ) {
				$this->add_error( sprintf(
					__( '%s is required.', 'limb-chatbot' ),
					$this->get_label( $parameter )
				) );

				return false;
			}

			// If empty and not required, it's valid
			return true;
		}

		// Validate that the value matches one of the allowed options
		return $this->validate_option_exists( $trimmed_value, $parameter );
	}

	/**
	 * Validate that the provided value is one of the allowed options
	 *
	 * @param  string  $value  Value to validate
	 * @param  Parameter  $parameter  Parameter configuration
	 *
	 * @return bool True if value is a valid option, false otherwise
	 * @since 1.0.0
	 */
	private function validate_option_exists( string $value, Parameter $parameter ): bool {
		$config = $parameter->get_config();

		if ( empty( $config['options'] ) || ! is_array( $config['options'] ) ) {
			$this->add_error( sprintf(
				__( 'No valid options configured for %s.', 'limb-chatbot' ),
				$this->get_label( $parameter )
			) );

			return false;
		}

		$value = trim( $value );
		$value_lower = strtolower( $value );
		foreach ( $config['options'] as $option ) {
			// Check if value matches option value (exact match)
			if ( ! empty( $option['value'] ) && strtolower( $option['value'] ) == $value_lower ) {
				return true;
			}

			// Check if value matches option label (exact match for emoji or case-insensitive for text)
			if ( ! empty( $option['label'] ) ) {
				// Exact match (handles emoji perfectly)
				if ( $option['label'] === $value ) {
					return true;
				}
				
				// Case-insensitive match (for text labels)
				if ( strtolower( $option['label'] ) === $value_lower ) {
					return true;
				}
			}
		}

		$this->add_error( sprintf(
			__( 'There is not such %s option.', 'limb-chatbot' ),
			strtolower( $this->get_label( $parameter ) ),

		) );

		return false;
	}

	/**
	 * @inheritDoc
	 */
	public function get_hint( Parameter $parameter ): ?string {
		$config = $parameter->get_config();

		if ( empty( $config['options'] ) || ! is_array( $config['options'] ) ) {
			return null;
		}

		$options_list = implode( ', ',
			array_map(
				function ( $option ) {
					return $option['label'] ?? $option['value'] ?? 'unknown';
				},
				$config['options']
			) );

		return sprintf(
			__( 'Available options: %s', 'limb-chatbot' ),
			$options_list
		);
	}

	public function get_config_schema(): array {
		return array(
			'options' => array(
				'type'        => 'array',
				'description' => 'Button options',
				'required'    => true,
			),
			'lead_capture_map_field' => array(
				'type'        => 'integer',
				'description' => 'Lead capture map field',
				'required'    => false,
			),
		);
	}
}

