<?php

namespace Limb_Chatbot\Includes\Services\Actions;

/**
 * Class Parameter_Value_Result
 *
 * Value object representing the result of parameter value processing.
 * Follows Single Responsibility Principle - encapsulates processing result.
 *
 * @package Limb_Chatbot\Includes\Services\Actions
 * @since 1.0.0
 */
class Parameter_Value_Result {

	/**
	 * Whether the processing was successful
	 *
	 * @var bool
	 */
	private bool $success;

	/**
	 * The processed value (if successful)
	 *
	 * @var string|null
	 */
	private ?string $value;

	/**
	 * Error message (if failed)
	 *
	 * @var string|null
	 */
	private ?string $error_message;

	/**
	 * All error messages (if multiple)
	 *
	 * @var array
	 */
	private array $errors;

	/**
	 * Constructor
	 *
	 * @param  bool  $success  Success status
	 * @param  string|null  $value  Processed value
	 * @param  string|null  $error_message  Error message
	 * @param  array  $errors  All errors
	 *
	 * @since 1.0.0
	 */
	private function __construct( bool $success, ?string $value, ?string $error_message, array $errors ) {
		$this->success       = $success;
		$this->value         = $value;
		$this->error_message = $error_message;
		$this->errors        = $errors;
	}

	/**
	 * Create a successful result
	 *
	 * @param  string  $value  The processed value
	 *
	 * @return self
	 * @since 1.0.0
	 */
	public static function success( string $value ): self {
		return new self( true, $value, null, [] );
	}

	/**
	 * Create a failed result
	 *
	 * @param  string  $error_message  Error message
	 * @param  array  $errors  All error messages
	 *
	 * @return self
	 * @since 1.0.0
	 */
	public static function failed( string $error_message, array $errors = [] ): self {
		return new self( false, null, $error_message, $errors ?: [ $error_message ] );
	}

	/**
	 * Check if processing was successful
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function is_success(): bool {
		return $this->success;
	}

	/**
	 * Check if processing failed
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function is_failed(): bool {
		return ! $this->success;
	}

	/**
	 * Get the processed value
	 *
	 * @return string|null
	 * @since 1.0.0
	 */
	public function get_value(): ?string {
		return $this->value;
	}

	/**
	 * Get the error message
	 *
	 * @return string|null
	 * @since 1.0.0
	 */
	public function get_error_message(): ?string {
		return $this->error_message;
	}

	/**
	 * Get all errors
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function get_errors(): array {
		return $this->errors;
	}
}

