<?php

namespace Limb_Chatbot\Includes\Repositories;

use Limb_Chatbot\Includes\Data_Objects\Wp_Object;
use Limb_Chatbot\Includes\Services\Data_Object_Collection;

class Wp_Special_Page_Object_Repository extends Wp_Object_Repository {
	/**
	 * Fetch special page objects (front page, blog home, etc.)
	 *
	 * @param $group_name
	 * @param $search_term
	 * @param $per_page
	 * @param $page
	 *
	 * @return Data_Object_Collection
	 */
	public function get_items( $group_name, $search_term, $per_page, $page ): Data_Object_Collection {
		$show_on_front = get_option( 'show_on_front', 'posts' );
		$front_page_id = get_option( 'page_on_front' );
		$blog_page_id = get_option( 'page_for_posts' );

		$special_pages = [];

		// Handle Homepage group
		if ( $group_name === 'homepage' ) {
			if ( $show_on_front === 'posts' ) {
				// Show "Latest posts" option
				$special_pages[] = [
					'id'          => null,
					'title'       => __( 'Latest posts', 'limb-chatbot' ),
					'link'        => home_url( '/' ),
					'identifier'  => 'page:front',
					'object_type' => 'special_page',
				];
			} else {
				// Show the selected static page
				$page_title = $front_page_id ? get_the_title( $front_page_id ) : __( '(no page selected)', 'limb-chatbot' );
				$special_pages[] = [
					'id'          => $front_page_id ? (int) $front_page_id : null,
					'title'       => $page_title,
					'link'        => home_url( '/' ),
					'identifier'  => 'page:front',
					'object_type' => 'special_page',
				];
			}
		}

		// Handle Posts Page group (only available when show_on_front is 'page')
		if ( $group_name === 'posts_page' && $show_on_front === 'page' ) {
			$page_title = $blog_page_id ? get_the_title( $blog_page_id ) : __( '(no page selected)', 'limb-chatbot' );
			$special_pages[] = [
				'id'          => $blog_page_id ? (int) $blog_page_id : null,
				'title'       => $page_title,
				'link'        => get_post_type_archive_link( 'post' ),
				'identifier'  => 'page:home',
				'object_type' => 'special_page',
			];
		}

		// Filter based on search
		if ( ! empty( $search_term ) ) {
			$special_pages = array_filter( $special_pages, function ( $page ) use ( $search_term ) {
				return str_contains( strtolower( $page['title'] ), strtolower( $search_term ) );
			} );
		}

		// Pagination logic
		$total_items   = count( $special_pages );
		$offset        = ( $page - 1 ) * $per_page;
		$special_pages = array_slice( $special_pages, $offset, $per_page );

		// Map to Wp_Object
		$objects = array_map( function ( $page_data ) {
			return Wp_Object::make( $page_data );
		}, $special_pages );

		// Create collection and set total
		$collection = new Data_Object_Collection( $objects );
		$collection->set_total( $total_items );

		// Add pagination properties
		$collection->push_property( 'page', $page );
		$collection->push_property( 'per_page', $per_page );
		$collection->push_property( 'total_pages', $total_items > 0 ? ceil( $total_items / $per_page ) : 0 );

		return $collection;
	}
}

