<?php

/**
 * @file This file is part of the PdfParser library.
 *
 * @author  Konrad Abicht <k.abicht@gmail.com>
 * @date    2021-02-09
 *
 * @license LGPLv3
 * @url     <https://github.com/smalot/pdfparser>
 *
 *  PdfParser is a pdf library written in PHP, extraction oriented.
 *  Copyright (C) 2017 - Sébastien MALOT <sebastien@malot.fr>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program.
 *  If not, see <http://www.pdfparser.org/sites/default/LICENSE.txt>.
 *
 * --------------------------------------------------------------------------------------
 *
 * About:
 * This file provides an alternative to the Composer-approach.
 * Include it into your project and all required files of PDFParser will be loaded automatically.
 * Please use it only, if Composer is not available.
 *
 * How to use:
 * 1. include this file as it is OR copy and rename it as you like (and then include it)
 * 2. afterwards you can use PDFParser classes
 * Done.
 */

// Check if pdfparser classes are already loaded by another plugin.
// Check for multiple key classes to ensure we catch if any plugin has loaded them.
// If any of these classes exist, skip loading to avoid conflicts.
if ( class_exists( 'Smalot\PdfParser\Parser', false ) ||
	 class_exists( 'Smalot\PdfParser\Element', false ) ||
	 class_exists( 'Smalot\PdfParser\PDFObject', false ) ||
	 class_exists( 'Smalot\PdfParser\Document', false ) ) {
	return;
}

/**
 * Loads all files found in a given folder.
 * Calls itself recursively for all sub folders.
 *
 * Uses a unique function name to avoid conflicts with other plugins.
 *
 * @param  string  $dir
 */
if ( ! function_exists( 'lbaic_require_pdfparser_files' ) ) {
	/**
	 * Recursively require all PHP files in a directory.
	 *
	 * @param string $dir Directory path.
	 */
	function lbaic_require_pdfparser_files( $dir ) {
		foreach ( new DirectoryIterator( $dir ) as $fileInfo ) {
			if ( $fileInfo->isDot() ) {
				continue;
			}

			if ( $fileInfo->isDir() ) {
				lbaic_require_pdfparser_files( $fileInfo->getPathname() );
			} elseif ( $fileInfo->getExtension() === 'php' ) {
				// Try to determine class name from file path
				$file_path = $fileInfo->getPathname();
				$relative_path = str_replace( __DIR__ . '/src/', '', $file_path );
				$relative_path = str_replace( '.php', '', $relative_path );
				$relative_path = str_replace( '/', '\\', $relative_path );
				$potential_class = 'Smalot\\PdfParser\\' . $relative_path;

				// Only require if class doesn't exist
				if ( ! class_exists( $potential_class, false ) && ! interface_exists( $potential_class, false ) ) {
					require_once $file_path;
				}
			}
		}
	}
}

$rootFolder = __DIR__ . '/src/Smalot/PdfParser';

// Manually require files, which can't be loaded automatically that easily.
// Check each class before requiring to avoid conflicts with other plugins.
if ( ! class_exists( 'Smalot\PdfParser\Element', false ) ) {
	require_once $rootFolder . '/Element.php';
}
if ( ! class_exists( 'Smalot\PdfParser\PDFObject', false ) ) {
	require_once $rootFolder . '/PDFObject.php';
}
if ( ! class_exists( 'Smalot\PdfParser\Font', false ) ) {
	require_once $rootFolder . '/Font.php';
}
if ( ! class_exists( 'Smalot\PdfParser\Page', false ) ) {
	require_once $rootFolder . '/Page.php';
}
if ( ! class_exists( 'Smalot\PdfParser\Element\ElementString', false ) ) {
	require_once $rootFolder . '/Element/ElementString.php';
}
if ( ! class_exists( 'Smalot\PdfParser\Encoding\AbstractEncoding', false ) ) {
	require_once $rootFolder . '/Encoding/AbstractEncoding.php';
}

/*
 * Load the rest of PDFParser files from /src/Smalot/PDFParser
 * Don't worry, it won't load files multiple times.
 */
lbaic_require_pdfparser_files( $rootFolder );
