<?php

namespace Limb_Chatbot\Includes\Interfaces;

/**
 * Interface Parameter_Type_Schema_Interface
 *
 * Contract for parameter type schema definitions.
 * Each parameter type handler should implement this to define its own schema.
 *
 * @package Limb_Chatbot\Includes\Interfaces
 * @since 1.0.0
 */
interface Parameter_Type_Schema_Interface {

	/**
	 * Get the validation rules schema for this parameter type.
	 *
	 * Returns array of allowed validation rules with their constraints.
	 *
	 * @return array {
	 *     @type string $rule_name => array {
	 *         @type string $type Type of value (string, integer, boolean, array)
	 *         @type string $description Human-readable description
	 *         @type bool $required Whether rule is required
	 *         @type mixed $default Default value if applicable
	 *         @type array $allowed Allowed values for enum
	 *         @type int $min_value Minimum numeric value
	 *         @type int $max_value Maximum numeric value
	 *     }
	 * }
	 * @since 1.0.0
	 */
	public function get_validation_rules_schema(): array;

	/**
	 * Get the configuration schema for this parameter type.
	 *
	 * Returns array of allowed configurations with their constraints.
	 *
	 * @return array {
	 *     @type string $config_name => array {
	 *         @type string $type Type of value
	 *         @type string $description Human-readable description
	 *         @type bool $required Whether config is required
	 *         @type mixed $default Default value
	 *         @type array $allowed Allowed values for enum
	 *         @type int $min_value Minimum numeric value
	 *         @type int $max_value Maximum numeric value
	 *     }
	 * }
	 * @since 1.0.0
	 */
	public function get_config_schema(): array;

	/**
	 * Check if a validation rule is valid for this type.
	 *
	 * @param  string  $rule_key    Rule key
	 * @param  mixed   $rule_value  Rule value
	 *
	 * @return bool True if valid
	 * @since 1.0.0
	 */
	public function is_valid_rule( string $rule_key, $rule_value ): bool;

	/**
	 * Check if a configuration is valid for this type.
	 *
	 * @param  string  $config_key    Config key
	 * @param  mixed   $config_value  Config value
	 *
	 * @return bool True if valid
	 * @since 1.0.0
	 */
	public function is_valid_config( string $config_key, $config_value ): bool;
}

