<?php

namespace Limb_Chatbot\Includes\Interfaces;

use Limb_Chatbot\Includes\Data_Objects\Parameter;

/**
 * Interface Parameter_Type_Interface
 *
 * Contract for parameter type handlers following Strategy Pattern.
 * Each parameter type (text, email, url, etc.) implements this interface
 * to handle its own validation and sanitization logic.
 *
 * This follows the Open/Closed Principle - new types can be added without
 * modifying existing code.
 *
 * @package Limb_Chatbot\Includes\Interfaces
 * @since 1.0.0
 */
interface Parameter_Type_Interface {

	/**
	 * Get the parameter type identifier this handler supports
	 *
	 * @return string The parameter type identifier (e.g., 'text', 'email', 'url')
	 * @since 1.0.0
	 */
	public function get_type(): string;

	/**
	 * Sanitize a parameter value
	 *
	 * Clean and normalize the input value before validation.
	 * Should always return a string, even if empty.
	 *
	 * @param  string  $value  The raw value to sanitize
	 * @param  Parameter  $parameter  The parameter configuration
	 *
	 * @return string The sanitized value
	 * @since 1.0.0
	 */
	public function sanitize( string $value, Parameter $parameter );

	/**
	 * Validate a parameter value
	 *
	 * Validate the sanitized value against parameter rules and type requirements.
	 * Errors should be stored internally and accessible via get_errors().
	 *
	 * @param  string  $value  The sanitized value to validate
	 * @param  Parameter  $parameter  The parameter configuration
	 *
	 * @return bool True if valid, false otherwise
	 * @since 1.0.0
	 */
	public function validate( string $value, Parameter $parameter ): bool;

	/**
	 * Get validation errors from the last validate() call
	 *
	 * @return array Array of error messages
	 * @since 1.0.0
	 */
	public function get_errors(): array;

	/**
	 * Get the first validation error
	 *
	 * @return string|null First error message or null if no errors
	 * @since 1.0.0
	 */
	public function get_first_error(): ?string;

	/**
	 * Check if there are validation errors
	 *
	 * @return bool True if errors exist
	 * @since 1.0.0
	 */
	public function has_errors(): bool;

	/**
	 * Clear all validation errors
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function clear_errors(): void;

	/**
	 * Get a helpful hint for this parameter type
	 *
	 * This method generates contextual hints to help users understand
	 * what format is expected. For example, email types might return
	 * "For example: user@example.com"
	 *
	 * @param  Parameter  $parameter  The parameter configuration
	 *
	 * @return string|null Hint message or null if no hint available
	 * @since 1.0.0
	 */
	public function get_hint( Parameter $parameter ): ?string;

	/**
	 * Check if this parameter type has post-collection steps
	 *
	 * Some parameter types may require additional steps after collection.
	 * For example, email verification requires a verification step.
	 *
	 * @param  Parameter  $parameter  The parameter configuration
	 *
	 * @return bool True if parameter type has post-collection steps
	 * @since 1.0.0
	 */
	public function has_post_collection_steps( Parameter $parameter ): bool;

	/**
	 * Get post-collection step type
	 *
	 * Returns the action plan step type for post-collection processing.
	 * For email verification, this would return TYPE_EMAIL_VERIFICATION.
	 *
	 * @param  Parameter  $parameter  The parameter configuration
	 *
	 * @return int|null The action plan step type, or null if no post-collection steps
	 * @since 1.0.0
	 */
	public function get_post_collection_step_type( Parameter $parameter ): ?int;

	/**
	 * Get post-collection step data
	 *
	 * Returns the data to attach to the post-collection action plan step.
	 * This data will be available when processing the step.
	 *
	 * @param  Parameter  $parameter  The parameter configuration
	 * @param  string  $collected_value  The value collected from user
	 *
	 * @return array Step data to attach to the action plan step
	 * @since 1.0.0
	 */
	public function get_post_collection_step_data( Parameter $parameter, string $collected_value ): array;

	/**
	 * Handle post-collection step processing
	 *
	 * This is called when processing a post-collection step for this parameter type.
	 * For email verification, this validates the verification code.
	 *
	 * @param  Parameter  $parameter  The parameter configuration
	 * @param  array  $step_data  The step data from action plan step
	 * @param  string  $user_input  The user's input (e.g., verification code)
	 *
	 * @return array Result with 'success' bool and optional 'message' string
	 * @since 1.0.0
	 */
	public function process_post_collection_step( Parameter $parameter, array $step_data, string $user_input );
}

