<?php

namespace Limb_Chatbot\Includes\Interfaces;

use Limb_Chatbot\Includes\Data_Objects\Chatbot_User;
use Limb_Chatbot\Includes\Services\Notifications\Notification_Envelope;

/**
 * Interface for registered notification types.
 *
 * Each type defines key, target roles, channels, and how to build envelope content
 * from context (e.g. new chat created). Role-based filtering is applied before sending.
 *
 * @package Limb_Chatbot\Includes\Interfaces
 * @since 1.0.15
 */
interface Notification_Type_Interface {

	/**
	 * Unique notification type key (e.g. 'chat_created', 'admin_alert').
	 *
	 * @return string
	 * @since 1.0.15
	 */
	public function get_key(): string;

	/**
	 * Roles that are allowed to receive this notification.
	 *
	 * Return array of role slugs: 'guest', 'user', 'agent', 'admin'.
	 * Empty array means no one receives it (disabled).
	 *
	 * @return string[]
	 * @since 1.0.15
	 */
	public function get_target_roles(): array;

	/**
	 * Channel keys through which this notification can be sent (e.g. ['email']).
	 *
	 * @return string[]
	 * @since 1.0.15
	 */
	public function get_channels(): array;

	/**
	 * Whether this notification should be sent instantly (true) or queued (false).
	 *
	 * Instant notifications are sent immediately when queue() is called; queued ones
	 * are scheduled via Action Scheduler (or do_action if AS is not available).
	 *
	 * @return bool True for instant, false for queued.
	 * @since 1.0.15
	 */
	public function is_instant(): bool;

	/**
	 * Build the notification envelope for the recipient from context payload.
	 *
	 * @param  Chatbot_User  $recipient  Target chatbot user.
	 * @param  array  $payload  Context data (e.g. ['chat' => Chat, 'chatbot' => Chatbot]).
	 *
	 * @return Notification_Envelope|null Envelope to send, or null to skip.
	 * @since 1.0.15
	 */
	public function build_envelope( Chatbot_User $recipient, array $payload ): ?Notification_Envelope;
}
