<?php

namespace Limb_Chatbot\Includes\Interfaces;

use Limb_Chatbot\Includes\Data_Objects\Job;
use Limb_Chatbot\Includes\Data_Objects\Task;
use Limb_Chatbot\Includes\Exceptions\Exception;

/**
 * Interface for queue job handlers.
 *
 * Defines the contract that all queue handler classes must implement.
 * Each handler is responsible for a specific job type's logic.
 *
 * @since 1.0.9
 */
interface Job_Handler_Interface {

	/**
	 * Validate job configuration before job creation.
	 *
	 * @param array $config Job configuration.
	 * @return bool True if valid.
	 * @throws Exception If validation fails.
	 * @since 1.0.9
	 */
	public function validate( array $config, ?string $chatbot_uuid = null ): bool;

	/**
	 * Get the total number of tasks that will be generated for this job.
	 *
	 * Called during job creation to determine the total task count without
	 * generating all tasks (important for large datasets).
	 *
	 * @param array $config Job configuration.
	 * @return int Total task count.
	 * @throws Exception If calculation fails.
	 * @since 1.0.9
	 */
	public function get_total( array $config, Job $job ): int;

	/**
	 * Generate a batch of tasks for the job (chunked generation).
	 *
	 * Called repeatedly to generate tasks in chunks until all are created.
	 * Essential for large datasets (e.g., 100k products) to prevent timeouts.
	 *
	 * @param Job $job Job instance.
	 * @param array $config Job configuration.
	 * @param int $offset Starting offset for this batch.
	 * @param int $limit Maximum number of tasks to generate in this batch.
	 * @return int Number of tasks actually created.
	 * @throws Exception If task generation fails.
	 * @since 1.0.9
	 */
	public function generate_task_batch( Job $job, array $config, int $offset, int $limit ): int;

	/**
	 * Process a single task.
	 *
	 * @param Task $task Task to process.
	 * @return bool True on success, false on failure.
	 * @throws Exception If processing fails.
	 * @since 1.0.9
	 */
	public function process_task( Task $task ): bool;

	/**
	 * Determine if an exception is critical and should pause the job.
	 *
	 * @param Exception $exception Exception that occurred.
	 * @return bool True if critical, false otherwise.
	 * @since 1.0.9
	 */
	public function is_critical_error( Exception $exception ): bool;

	/**
	 * Get the job type this handler manages.
	 *
	 * @return string Job type identifier.
	 * @since 1.0.9
	 */
	public function get_job_type(): string;
}

