<?php

namespace Limb_Chatbot\Includes\Integrations\Telegram\Utilities;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Integrations\Telegram\Endpoints\Messages\Messages_Endpoint;
use Limb_Chatbot\Includes\Integrations\Telegram\Telegram;

/**
 * Telegram Messages Utility
 *
 * Utility for Telegram message and forum topic operations.
 * Wraps the Messages_Endpoint with context-aware configuration.
 *
 * @package Limb_Chatbot\Includes\Integrations\Telegram\Utilities
 * @since 1.0.11
 */
class Messages_Utility {

	/**
	 * API endpoint handler for message operations.
	 *
	 * @var Messages_Endpoint
	 * @since 1.0.11
	 */
	public Messages_Endpoint $endpoint;

	/**
	 * Telegram configuration object.
	 *
	 * @var Config|null
	 * @since 1.0.11
	 */
	public ?Config $config = null;

	/**
	 * HTTP request timeout in seconds.
	 *
	 * @var int
	 * @since 1.0.11
	 */
	public int $timeout = 30;

	/**
	 * Constructor.
	 *
	 * Initializes the utility with config and instantiates the related API endpoint.
	 *
	 * @param Config|null $config Configuration object.
	 *
	 * @since 1.0.11
	 */
	public function __construct( ?Config $config = null ) {
		if ( is_null( $config ) ) {
			$this->config = Config::where( [
				'default'    => true,
				'related_to' => Telegram::class,
			] )->first();
		} else {
			$this->config = $config;
		}

		// Initialize endpoint
		$this->endpoint = new Messages_Endpoint( $this );
	}

	/**
	 * Send a message to a chat or topic.
	 *
	 * @param string|int  $chat_id           Chat ID.
	 * @param string      $text              Message text.
	 * @param int|null    $message_thread_id Optional topic thread ID.
	 * @param string|null $parse_mode        Optional parse mode.
	 *
	 * @return array Message data.
	 * @since 1.0.11
	 */
	public function send_message( $chat_id, string $text, ?int $message_thread_id = null, ?string $parse_mode = 'HTML' ): array {
		return $this->endpoint->send_message( $chat_id, $text, $message_thread_id, $parse_mode );
	}

	/**
	 * Get updates from Telegram.
	 *
	 * @param int|null $offset  Offset for pagination.
	 * @param int      $limit   Maximum number of updates.
	 * @param int      $timeout Timeout for long polling.
	 *
	 * @return array Array of update objects.
	 * @since 1.0.11
	 */
	public function get_updates( ?int $offset = null, int $limit = 100, int $timeout = 0 ): array {
		return $this->endpoint->get_updates( $offset, $limit, $timeout );
	}

	/**
	 * Close a forum topic.
	 *
	 * @param string|int $chat_id           Chat ID.
	 * @param int        $message_thread_id Topic thread ID.
	 *
	 * @return bool True on success.
	 * @since 1.0.11
	 */
	public function close_forum_topic( $chat_id, int $message_thread_id ): bool {
		return $this->endpoint->close_forum_topic( $chat_id, $message_thread_id );
	}

	/**
	 * Reopen a closed forum topic.
	 *
	 * @param string|int $chat_id           Chat ID.
	 * @param int        $message_thread_id Topic thread ID.
	 *
	 * @return bool True on success.
	 * @since 1.0.11
	 */
	public function reopen_forum_topic( $chat_id, int $message_thread_id ): bool {
		return $this->endpoint->reopen_forum_topic( $chat_id, $message_thread_id );
	}

	/**
	 * Get user profile photos.
	 *
	 * @param int $user_id Telegram user ID.
	 * @param int $offset  Photo offset.
	 * @param int $limit   Max number of photos.
	 *
	 * @return array UserProfilePhotos object.
	 * @since 1.0.11
	 */
	public function get_user_profile_photos( int $user_id, int $offset = 0, int $limit = 1 ): array {
		return $this->endpoint->get_user_profile_photos( $user_id, $offset, $limit );
	}

	/**
	 * Get user avatar URL.
	 *
	 * @param int $user_id Telegram user ID.
	 *
	 * @return string|null Avatar URL or null if no photo.
	 * @since 1.0.11
	 */
	public function get_user_avatar_url( int $user_id ): ?string {
		return $this->endpoint->get_user_avatar_url( $user_id );
	}

	/**
	 * Get configuration object.
	 *
	 * @return Config|null
	 * @since 1.0.11
	 */
	public function get_config(): ?Config {
		return $this->config;
	}

	/**
	 * Get timeout value.
	 *
	 * @return int
	 * @since 1.0.11
	 */
	public function get_timeout(): int {
		return $this->timeout;
	}
}

