<?php

namespace Limb_Chatbot\Includes\Integrations\Slack\Utilities;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Integrations\Slack\Endpoints\Channels\Channels_Endpoint;
use Limb_Chatbot\Includes\Integrations\Slack\Slack;
use Limb_Chatbot\Includes\Utilities\Slack_Channels_Utility as Global_Utility;

/**
 * Slack Channels Utility
 *
 * Provider-specific utility for Slack channel operations.
 * Wraps the Channels_Endpoint with context-aware configuration.
 *
 * @package Limb_Chatbot\Includes\Integrations\Slack\Utilities
 * @since 1.0.0
 */
class Channels_Utility {

	/**
	 * Global utility instance holding context.
	 *
	 * @var Global_Utility
	 * @since 1.0.0
	 */
	public Global_Utility $global_utility;

	/**
	 * API endpoint handler for channel operations.
	 *
	 * @var Channels_Endpoint
	 * @since 1.0.0
	 */
	public Channels_Endpoint $endpoint;

	/**
	 * Slack configuration object.
	 *
	 * @var Config|null
	 * @since 1.0.0
	 */
	public ?Config $config = null;

	/**
	 * HTTP request timeout in seconds.
	 *
	 * @var int
	 * @since 1.0.0
	 */
	public int $timeout = 30;

	/**
	 * Channel types to fetch.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $types = 'public_channel,private_channel';

	/**
	 * Whether to exclude archived channels.
	 *
	 * @var bool
	 * @since 1.0.0
	 */
	public bool $exclude_archived = true;

	/**
	 * Specific channel ID for single channel operations.
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $channel_id = null;

	/**
	 * Constructor.
	 *
	 * Initializes the utility with global context, loads the config if not set,
	 * and instantiates the related API endpoint.
	 *
	 * @param Global_Utility $global_utility Global utility object containing config.
	 *
	 * @since 1.0.0
	 */
	public function __construct( Global_Utility $global_utility ) {
		$this->global_utility = $global_utility;
		// Load default config if not set
		if ( is_null( $this->global_utility->config ) ) {
			$this->config = Config::where( [
				'default'    => true,
				'related_to' => Slack::class,
			] )->first();
		} else {
			$this->config = $this->global_utility->config;
		}

		// Copy properties from global utility
		$this->types            = $this->global_utility->types;
		$this->exclude_archived = $this->global_utility->exclude_archived;
		$this->channel_id       = $this->global_utility->channel_id;
		$this->timeout          = $this->global_utility->get_timeout();

		// Initialize endpoint
		$this->endpoint = new Channels_Endpoint( $this );
	}

	/**
	 * List all channels.
	 *
	 * @return array Array of channel objects.
	 * @since 1.0.0
	 */
	public function list() {
		return $this->endpoint->list();
	}

	/**
	 * Retrieve a single channel.
	 *
	 * @return array Channel object.
	 * @since 1.0.0
	 */
	public function retrieve() {
		return $this->endpoint->retrieve();
	}

	/**
	 * Get configuration object.
	 *
	 * @return Config|null
	 * @since 1.0.0
	 */
	public function get_config(): ?Config {
		return $this->config;
	}

	/**
	 * Get timeout value.
	 *
	 * @return int
	 * @since 1.0.0
	 */
	public function get_timeout(): int {
		return $this->timeout;
	}
}

