<?php

namespace Limb_Chatbot\Includes\Integrations\Slack\Endpoints\Messages;

use Limb_Chatbot\Includes\Integrations\Slack\Handlers\Response_Handler;

/**
 * Messages Response Handler
 *
 * Specialized response handler for Slack message and channel endpoints.
 * Handles response parsing and formatting for channel creation, invites, and messages.
 *
 * @package Limb_Chatbot\Includes\Integrations\Slack\Endpoints\Handlers
 * @since 1.0.0
 */
class Messages_Response_Handler extends Response_Handler {

	/**
	 * Get formatted channel from response.
	 *
	 * @return array Formatted channel object.
	 * @since 1.0.0
	 */
	public function get_channel(): array {
		$body    = $this->get_body();
		$channel = $body['channel'] ?? [];

		return $this->format_channel( $channel );
	}

	/**
	 * Get formatted message from response.
	 *
	 * @return array Formatted message object.
	 * @since 1.0.0
	 */
	public function get_message(): array {
		$body    = $this->get_body();
		$message = $body['message'] ?? [];

		return $this->format_message( $message );
	}

	/**
	 * Format a channel for consistent output.
	 *
	 * @param array $channel Raw channel from Slack API.
	 *
	 * @return array Formatted channel.
	 * @since 1.0.0
	 */
	private function format_channel( array $channel ): array {
		return [
			'id'         => $channel['id'] ?? '',
			'name'       => $channel['name'] ?? '',
			'is_channel' => $channel['is_channel'] ?? false,
			'is_private' => $channel['is_private'] ?? false,
			'created'    => $channel['created'] ?? 0,
			'creator'    => $channel['creator'] ?? '',
		];
	}

	/**
	 * Format a message for consistent output.
	 *
	 * @param array $message Raw message from Slack API.
	 *
	 * @return array Formatted message.
	 * @since 1.0.0
	 */
	private function format_message( array $message ): array {
		return [
			'ts'      => $message['ts'] ?? '',
			'channel' => $message['channel'] ?? '',
			'text'    => $message['text'] ?? '',
			'user'    => $message['user'] ?? '',
		];
	}
}

