<?php

namespace Limb_Chatbot\Includes\Integrations\Slack\Endpoints;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Services\Endpoint;

/**
 * Base Slack Endpoint
 *
 * Provides common functionality for Slack API interactions including
 * authentication header generation and base URL management.
 *
 * @package Limb_Chatbot\Includes\Integrations\Slack\Endpoints
 * @since 1.0.0
 */
class Slack_Endpoint extends Endpoint {

	/**
	 * Slack API base URL.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	const API_BASE_URL = 'https://slack.com/api';

	/**
	 * Generate HTTP headers for Slack API requests.
	 *
	 * @param Config|null $config Optional Config object. If null, uses internal config.
	 *
	 * @return array HTTP headers including Authorization.
	 * @throws Exception If bot token is missing or empty.
	 * @since 1.0.0
	 */
	protected function get_header( ?Config $config = null ): array {
		$config = $config ?? $this->get_config();
		if ( ! ( $config instanceof Config ) ) {
			throw new Exception(
				Error_Codes::AUTHENTICATION_API_KEY_MISSING,
				__( 'Slack configuration is missing.', 'limb-chatbot' ),
				null,
				400
			);
		}

		$params = $config->get_params();
		if ( empty( $params['bot_token'] ) ) {
			throw new Exception(
				Error_Codes::AUTHENTICATION_API_KEY_MISSING,
				__( 'Slack bot token is missing or empty.', 'limb-chatbot' ),
				null,
				400
			);
		}

		return array(
			'Authorization' => 'Bearer ' . $params['bot_token'],
			'Content-Type'  => 'application/json',
		);
	}

	/**
	 * Get configuration object from utility.
	 *
	 * @return Config|null
	 * @since 1.0.0
	 */
	public function get_config(): ?Config {
		return $this->utility->config ?? null;
	}
}

