<?php

namespace Limb_Chatbot\Includes\Frontend;

use Limb_Chatbot\Includes\Traits\SingletonTrait;

class Page_Context {

	use SingletonTrait;


	/**
	 * Collect the current page context.
	 *
	 * @return array
	 */
	public static function get_context(): array {
		$context = [
			'type'       => 'unknown',
			'id'         => null,
			'identifier' => 'unknown',
		];

		$queried = get_queried_object();
		$front_page_id = (int) get_option( 'page_on_front' );
		$blog_page_id = (int) get_option( 'page_for_posts' );

		if ( is_front_page() ) {
			$context['type']       = 'front';
			$context['id']         = get_queried_object_id();
			$context['identifier'] = 'page:front';
			// If it's a static front page, also include the page identifier
			if ( $front_page_id ) {
				$context['post_type'] = 'page';
				$context['generic_identifiers'] = [ "singular:page:{$front_page_id}", "singular:page" ];
			}
		} elseif ( is_home() ) {
			$context['type']       = 'home';
			$context['id']         = $blog_page_id;
			$context['identifier'] = 'page:home';
			// If it's a static posts page, also include the page identifier
			if ( $blog_page_id ) {
				$context['post_type'] = 'page';
				$context['generic_identifiers'] = [ "singular:page:{$blog_page_id}", "singular:page" ];
			}
		} elseif ( is_singular() ) {
			$post_type             = get_post_type();
			$post_id               = get_queried_object_id();
			$context['type']       = 'singular';
			$context['id']         = $post_id;
			$context['post_type']  = $post_type;

			// Check if this page is actually the front page or posts page
			if ( $post_id === $front_page_id ) {
				$context['identifier'] = 'page:front';
				$context['generic_identifiers'] = [ "singular:{$post_type}:{$post_id}", "singular:{$post_type}" ];
			} elseif ( $post_id === $blog_page_id ) {
				$context['identifier'] = 'page:home';
				$context['generic_identifiers'] = [ "singular:{$post_type}:{$post_id}", "singular:{$post_type}" ];
			} else {
				$context['identifier'] = "singular:{$post_type}:{$post_id}";
				// Also add generic singular identifier for this post type
				$context['generic_identifiers'] = [ "singular:{$post_type}", ];
			}
		} elseif ( is_post_type_archive() ) {
			$post_type             = get_post_type();
			$context['type']       = 'cpt_archive';
			$context['post_type']  = $post_type;
			$context['identifier'] = "archive:{$post_type}";
		} elseif ( is_tax() || is_category() || is_tag() ) {
			$context['type']       = 'term';
			$context['id']         = $queried->term_id ?? null;
			$context['taxonomy']   = $queried->taxonomy ?? null;
			$context['identifier'] = "term:{$context['taxonomy']}:{$context['id']}";
			// Also add generic taxonomy identifier
			$context['generic_identifiers'] = [ "term:{$context['taxonomy']}", ];
		} elseif ( is_archive() ) {
			$post_type             = get_post_type();
			$context['type']       = 'archive';
			$context['post_type']  = $post_type;
			$context['identifier'] = "archive:{$post_type}";
		}

		return $context;
	}


	/**
	 * Check if a chatbot should be shown based on show_in and hide_in settings.
	 *
	 * @param array $show_in Array of page identifiers where chatbot should show
	 * @param array $hide_in Array of page identifiers where chatbot should hide
	 * @param array $context Current page context from get_context()
	 *
	 * @return bool True if chatbot should be shown, false otherwise
	 */
	public static function should_show_chatbot( $show_in, $hide_in, $context ): bool {
		// Get the identifier and any generic identifiers
		$identifier           = $context['identifier'] ?? 'unknown';
		$generic_identifiers  = $context['generic_identifiers'] ?? [];
		$all_identifiers      = array_merge( [ $identifier ], $generic_identifiers );

		// Default behavior if both are empty
		if ( empty( $show_in ) && empty( $hide_in ) ) {
			return true;
		}

		// If show_in contains 'all', check hide_in for exceptions
		if ( in_array( 'all', $show_in, true ) ) {
			// Check if current page is in hide_in list
			foreach ( $all_identifiers as $id ) {
				if ( in_array( $id, $hide_in, true ) ) {
					return false;
				}
			}
			return true;
		}

		// If hide_in contains 'all', check show_in for exceptions
		if ( in_array( 'all', $hide_in, true ) ) {
			// Check if current page is in show_in list
			foreach ( $all_identifiers as $id ) {
				if ( in_array( $id, $show_in, true ) ) {
					return true;
				}
			}
			return false;
		}

		// Neither contains 'all', so check if current page is explicitly in show_in
		foreach ( $all_identifiers as $id ) {
			if ( in_array( $id, $show_in, true ) ) {
				// But also check if it's explicitly hidden
				if ( in_array( $id, $hide_in, true ) ) {
					return false;
				}
				return true;
			}
		}

		// Not explicitly shown, so hide by default
		return false;
	}
}