<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Dataset;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Source_Fetcher_Interface;
use Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers\CPT_Source_Fetcher;
use Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers\File_Source_Fetcher;
use Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers\Manual_Source_Fetcher;
use Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers\Posts_Source_Fetcher;
use Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers\Terms_Source_Fetcher;
use Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers\Text_Source_Fetcher;
use Limb_Chatbot\Includes\Services\Knowledge\Source_Object_Fetchers\URL_Source_Fetcher;

/**
 * Class Source_Fetcher_Factory
 *
 * Factory responsible for creating dataset source fetcher instances
 * based on the given source type.
 *
 * Supported sources:
 *  - Posts
 *  - Terms
 *  - Manual entries
 *
 * @package Limb_Chatbot\Includes\Factories
 * @since 1.0.0
 */
class Source_Fetcher_Factory {

	/**
	 * Create a source fetcher instance for the given dataset source type.
	 *
	 * @param  string  $source_type  The type of dataset source.
	 *
	 * @return Source_Fetcher_Interface The appropriate source fetcher instance.
	 *
	 * @throws Exception If the given source type is not supported.
	 *
	 * @since 1.0.0
	 */
	public function make( string $source_type ): Source_Fetcher_Interface {
		switch ( $source_type ) {
			case Dataset::SOURCE_POST:
				return new Posts_Source_Fetcher();
			case Dataset::SOURCE_TERM:
				return new Terms_Source_Fetcher();
			case Dataset::SOURCE_QA:
				return new Manual_Source_Fetcher();
			case Dataset::SOURCE_FILE:
				return new File_Source_Fetcher();
			case Dataset::SOURCE_TEXT:
				return new Text_Source_Fetcher();
			case Dataset::SOURCE_CPT:
				return new CPT_Source_Fetcher();
			case Dataset::SOURCE_URL:
				return new URL_Source_Fetcher();
			default:
				throw new Exception( Error_Codes::VALIDATION_INVALID_VALUE,
					__( 'Invalid source type.', 'limb-chatbot' ) );
		}
	}
}