<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Parameter;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Parameter_Type_Interface;
use Limb_Chatbot\Includes\Services\Actions\Parameter_Types\Buttons_Parameter_Type;
use Limb_Chatbot\Includes\Services\Actions\Parameter_Types\Email_Parameter_Type;
use Limb_Chatbot\Includes\Services\Actions\Parameter_Types\Text_Parameter_Type;
use Limb_Chatbot\Includes\Services\Actions\Parameter_Types\Textarea_Parameter_Type;
use Limb_Chatbot\Includes\Services\Actions\Parameter_Types\Url_Parameter_Type;
use Limb_Chatbot\Includes\Services\Actions\Parameter_Types\Vector_Search_Parameter_Type;

/**
 * Class Parameter_Type_Factory
 *
 * Factory for creating and retrieving parameter type handlers.
 * Implements Service Locator pattern for parameter type strategies.
 *
 * Allows for dynamic extension of parameter type handling without
 * modifying core action processing logic.
 *
 * @package Limb_Chatbot\Includes\Factories
 * @since 1.0.0
 */
class Parameter_Type_Factory {

	/**
	 * Get parameter type handler for a specific type
	 *
	 * @param  string  $type  Parameter type identifier (e.g., 'email', 'text', 'url')
	 *
	 * @return Parameter_Type_Interface Parameter type handler
	 * @throws \Exception If parameter type is not supported
	 * @since 1.0.0
	 */
	public function make( string $type ): Parameter_Type_Interface {
		switch ( $type ) {
			case Parameter::TYPE_EMAIL:
				$handler = new Email_Parameter_Type();
				break;
			case Parameter::TYPE_TEXT:
				$handler = new Text_Parameter_Type();
				break;
			case Parameter::TYPE_TEXTAREA:
				$handler = new Textarea_Parameter_Type();
				break;
			case Parameter::TYPE_URL:
				$handler = new Url_Parameter_Type();
				break;
			case Parameter::TYPE_BUTTONS:
				$handler = new Buttons_Parameter_Type();
				break;
			case Parameter::TYPE_VECTOR_SEARCH:
				$handler = new Vector_Search_Parameter_Type();
				break;
			default:
				throw new Exception( Error_Codes::NOT_SUPPORTED,
					sprintf(
						__( 'Unsupported parameter type: %s', 'limb-chatbot' ),
						$type
					)
				);
		}

		// Cache the instance for reuse
		return $handler;
	}

}
