<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Parameter;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Parameter_Callback_Handler_Interface;
use Limb_Chatbot\Includes\Services\Actions\Vector_Search_Handler;

/**
 * Factory for creating parameter callback handlers based on parameter type.
 *
 * Implements the Factory pattern to instantiate appropriate handler classes
 * for different parameter callback types.
 *
 * @since 1.0.0
 */
class Parameter_Callback_Handler_Factory {

	/**
	 * Create a parameter callback handler for the given parameter.
	 *
	 * Resolves the appropriate handler based on parameter type.
	 * Supports TYPE_VECTOR_SEARCH and other parameter types.
	 *
	 * @param Parameter $parameter The parameter instance.
	 *
	 * @return Parameter_Callback_Handler_Interface Handler instance.
	 * @throws Exception When parameter type is not supported.
	 *
	 * @since 1.0.0
	 */
	public function make( Parameter $parameter ): Parameter_Callback_Handler_Interface {
		$parameter_type = $parameter->get_type();

		if ( Parameter::TYPE_VECTOR_SEARCH === $parameter_type ) {
			return new Vector_Search_Handler();
		}

		throw new Exception(
			Error_Codes::NOT_SUPPORTED,
			sprintf(
				__( 'Parameter callback handler not supported for type: %s', 'limb-chatbot' ),
				$parameter_type
			)
		);
	}
}

