<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Chat;
use Limb_Chatbot\Includes\Data_Objects\Chatbot;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Repositories\Message_Repository;
use Limb_Chatbot\Includes\Services\Message_Service;


/**
 * Factory to create appropriate Message_Service instances based on chat and chatbot type.
 *
 * @since 1.0.0
 */
class Message_Service_Factory {

	/**
	 * Create and return a Message_Service instance.
	 *
	 * @param Chat $chat The chat data object.
	 *
	 * @return Message_Service
	 *
	 * @throws Exception If assistant service is missing for the AI provider.
	 *
	 * @since 1.0.0
	 */
	public function make( Chat $chat ): Message_Service {
		$chatbot    = $chat->get_chatbot();
		$repository = new Message_Repository();
		if ( $chatbot && $chatbot->get_type() === Chatbot::TYPE_ASSISTANT ) {
			$class = "Limb_Chatbot\\Includes\\AI_Providers\\{$chatbot->get_ai_provider()::$name}\\Services\\Message_Assistant_Service";
			if ( class_exists( $class ) ) {
				return new $class( $chat, $repository );
			}
			throw new Exception( Error_Codes::NOT_SUPPORTED, __( 'Assistant is missing for the provider.', 'limb-chatbot' ) );
		}

		return new Message_Service( $chat, $repository );
	}
}