<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Dataset;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Knowledge_Mapper_Interface;
use Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers\CPT_Knowledge_Mapper;
use Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers\File_Knowledge_Mapper;
use Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers\Manual_Knowledge_Mapper;
use Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers\Post_Knowledge_Mapper;
use Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers\Term_Knowledge_Mapper;
use Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers\Text_Knowledge_Mapper;
use Limb_Chatbot\Includes\Services\Knowledge\Knowledge_Mappers\URL_Knowledge_Mapper;

/**
 * Factory class for creating knowledge mapper instances based on source type.
 *
 * Provides the correct implementation of Knowledge_Mapper_Interface depending
 * on whether the source is a post, term, or manual entry.
 *
 * @since 1.0.0
 */
class Knowledge_Mapper_Factory {

	/**
	 * Create a knowledge mapper instance for the given source type.
	 *
	 * @param  string  $source  The type of source ('post', 'term', 'q_a').
	 *
	 * @return Knowledge_Mapper_Interface
	 *
	 * @throws Exception If the provided source type is invalid.
	 *
	 * @since 1.0.0
	 */
	public function make( $source ): Knowledge_Mapper_Interface {
		if ( $source == Dataset::SOURCE_POST ) {
			return new Post_Knowledge_Mapper();
		} elseif ( $source == Dataset::SOURCE_TERM ) {
			return new Term_Knowledge_Mapper();
		} elseif ( $source == Dataset::SOURCE_QA ) {
			return new Manual_Knowledge_Mapper();
		} elseif ( $source === Dataset::SOURCE_FILE ) {
			return new File_Knowledge_Mapper();
		} elseif ( $source === Dataset::SOURCE_TEXT ) {
			return new Text_Knowledge_Mapper();
		} elseif ( $source === Dataset::SOURCE_CPT ) {
			return new CPT_Knowledge_Mapper();
		} elseif ( $source === Dataset::SOURCE_URL ) {
			return new URL_Knowledge_Mapper();
		}
		throw new Exception( Error_Codes::VALIDATION_INVALID_VALUE, __( 'Invalid source type.', 'limb-chatbot' ) );
	}
}