<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Job;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Job_Handler_Interface;
use Limb_Chatbot\Includes\Services\Jobs\Handlers\Dataset_Delete;
use Limb_Chatbot\Includes\Services\Jobs\Handlers\Dataset_Generating;
use Limb_Chatbot\Includes\Services\Jobs\Handlers\Dataset_Regenerate;
use Limb_Chatbot\Includes\Services\Jobs\Handlers\Dataset_Sync;
use Limb_Chatbot\Includes\Services\Jobs\Handlers\File_Upload;
use Limb_Chatbot\Includes\Services\Jobs\Handlers\Sitemap_Scrape;

/**
 * Job Handler Factory
 *
 * Creates job handler instances based on job type.
 *
 * @since 1.0.9
 */
class Job_Handler_Factory {

	/**
	 * Create a handler for the given job type.
	 *
	 * @param  string  $job_type  Job type.
	 *
	 * @return Job_Handler_Interface Handler instance.
	 * @throws Exception If handler not found for job type.
	 * @since 1.0.9
	 */
	public function make( string $job_type ): Job_Handler_Interface {
		switch ( $job_type ) {
			case Job::TYPE_DATASET_GENERATING:
				return new Dataset_Generating();

			case Job::TYPE_DATASET_DELETE:
				return new Dataset_Delete();

			case Job::TYPE_DATASET_SYNC:
				return new Dataset_Sync();

			case Job::TYPE_DATASET_REGENERATE:
				return new Dataset_Regenerate();

		case Job::TYPE_FILE_UPLOAD:
			return new File_Upload();

		case Job::TYPE_SITEMAP_SCRAPE:
			return new Sitemap_Scrape();

		default:
			throw new Exception(
				'invalid_job_type',
				sprintf( __( 'No handler found for job type: %s', 'limb-chatbot' ), $job_type )
			);
		}
	}

	/**
	 * Check if a handler exists for the given job type.
	 *
	 * @param  string  $job_type  Job type.
	 *
	 * @return bool True if handler exists.
	 * @since 1.0.9
	 */
	public function has_handler( string $job_type ): bool {
		return in_array( $job_type, $this->get_supported_types() );
	}

	/**
	 * Get all supported job types.
	 *
	 * @return array Array of supported job types.
	 * @since 1.0.9
	 */
	public function get_supported_types(): array {
		return [
			Job::TYPE_DATASET_GENERATING,
			Job::TYPE_DATASET_DELETE,
			Job::TYPE_DATASET_SYNC,
			Job::TYPE_DATASET_REGENERATE,
			Job::TYPE_FILE_UPLOAD,
			Job::TYPE_SITEMAP_SCRAPE,
		];
	}
}

