<?php

namespace Limb_Chatbot\Factories;

use Limb_Chatbot\Includes\AI_Providers\AI_Providers;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Interfaces\Fine_Tuning_File_Formatting_Interface;


/**
 * Factory class to instantiate fine tuning file formatting services
 * based on the AI provider of the given model.
 *
 * @since 1.0.0
 */
class Fine_Tuning_File_Formatting_Service_Factory {

	/**
	 * Create a fine tuning file formatting service instance for the specified model.
	 *
	 * @param object $model Model object with method get_ai_provider_id().
	 *
	 * @return Fine_Tuning_File_Formatting_Interface
	 *
	 * @throws Exception If AI provider is missing or not supported.
	 *
	 * @since 1.0.0
	 */
	public function make( $model ): Fine_Tuning_File_Formatting_Interface {
		if ( ! ( $ai_provider = AI_Providers::instance()->get_ai_provider( $model->get_ai_provider_id() ) ) ) {
			throw new Exception( Error_Codes::AI_PROVIDER_NOT_SET, __( 'AI provider is missing.', 'limb-chatbot' ) );
		}
		$service = "Limb_Chatbot\\Includes\\AI_Providers\\{$ai_provider->get_name()}\\Services\\Fine_Tuning_File_Formatting_Service";
		if ( ! class_exists( $service ) ) {
			throw new Exception( Error_Codes::NOT_SUPPORTED, __( 'The ai_provider does not support fine tuning', 'limb-chatbot' ) );
		}

		return new $service();
	}
}