<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Dataset;
use Limb_Chatbot\Includes\Data_Schemas\Dataset_Csv_File;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\File_Reader_Interface;
use Limb_Chatbot\Includes\Services\Csv_File_Reader;
use Limb_Chatbot\Includes\Services\Helper;


/**
 * Factory class for creating file reader instances based on file type and arguments.
 *
 * @since 1.0.0
 */
class File_Reader_Factory {

	/**
	 * Create a file reader instance for a given file path and configuration arguments.
	 *
	 * @param string $file_path Absolute or relative path to the file.
	 * @param array  $args      Additional arguments used for reader configuration.
	 *
	 * @return File_Reader_Interface Instance of a file reader.
	 *
	 * @throws Exception If the file does not exist or if the file type is unsupported.
	 *
	 * @since 1.0.0
	 */
	public function make( string $file_path, array $args ): File_Reader_Interface {
		$file_path = Helper::get_wp_uploaded_file_dir( $file_path );
		if ( ! file_exists( $file_path ) ) {
			throw new Exception( Error_Codes::DATASET_FILE_IS_MISSING, __( 'Dataset file not found.', 'limb-chatbot' ) );
		}
		if ( str_ends_with( $file_path, '.csv' ) ) {
			if ( $args['model'] && $args['model'] == Dataset::class ) {
				$args['header_columns'] = Dataset_Csv_File::REQUIRED_COLUMNS;
			}

			return new Csv_File_Reader( $file_path, $args );
		}
		throw new Exception( Error_Codes::TECHNICAL_ERROR, __( 'Technical error.', 'limb-chatbot' ) );
	}
}