<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Interfaces\Chunk_Stringifier_Interface;
use Limb_Chatbot\Includes\Services\Knowledge\Stringifiers\Chunk_Stringifier;
use Limb_Chatbot\Includes\Services\Knowledge\Stringifiers\QA_Stringifier;

/**
 * Entry Stringifier Factory
 *
 * Factory for creating appropriate stringifiers based on entry type.
 * Follows Factory pattern for dependency management.
 *
 * @since 1.2.1
 */
class Entry_Stringifier_Factory {

	/**
	 * Chunk stringifier instance (singleton).
	 *
	 * @var Chunk_Stringifier|null
	 */
	private static ?Chunk_Stringifier $chunk_stringifier = null;

	/**
	 * QA stringifier instance (singleton).
	 *
	 * @var QA_Stringifier|null
	 */
	private static ?QA_Stringifier $qa_stringifier = null;

	/**
	 * Create appropriate stringifier for entry type.
	 *
	 * @param string $entry_type Entry type ('chunk' or 'qa').
	 * @return Chunk_Stringifier_Interface Stringifier instance.
	 * @since 1.2.1
	 */
	public function make( string $entry_type ): Chunk_Stringifier_Interface {
		if ( $entry_type === 'qa' ) {
			return $this->make_qa_stringifier();
		}

		return $this->make_chunk_stringifier();
	}

	/**
	 * Create chunk stringifier instance.
	 *
	 * @return Chunk_Stringifier Chunk stringifier.
	 * @since 1.2.1
	 */
	public function make_chunk_stringifier(): Chunk_Stringifier {
		if ( self::$chunk_stringifier === null ) {
			self::$chunk_stringifier = new Chunk_Stringifier();
		}

		return self::$chunk_stringifier;
	}

	/**
	 * Create QA stringifier instance.
	 *
	 * @return QA_Stringifier QA stringifier.
	 * @since 1.2.1
	 */
	public function make_qa_stringifier(): QA_Stringifier {
		if ( self::$qa_stringifier === null ) {
			self::$qa_stringifier = new QA_Stringifier();
		}

		return self::$qa_stringifier;
	}
}

