<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Chatbot;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Services\Chatbot_Service;
use Limb_Chatbot\Includes\Utilities\Chatbot_Utility;


/**
 * Factory for creating chatbot service instances based on chatbot type and AI provider.
 *
 * @since 1.0.0
 */
class Chatbot_Service_Factory {

	/**
	 * Creates an appropriate chatbot service instance for the given chatbot.
	 *
	 * @param Chatbot $chatbot Chatbot data object.
	 * @return mixed An instance of chatbot service.
	 *
	 * @throws Exception Throws exception if chatbot type or AI provider is unsupported.
	 *
	 * @since 1.0.0
	 */
	public function make( Chatbot $chatbot ) {
		$utility    = new Chatbot_Utility( $chatbot );
		if ( $utility->get_type() === Chatbot::TYPE_SIMPLE ) {
			return new Chatbot_Service( $utility );
		} elseif ( $utility->get_type() === Chatbot::TYPE_ASSISTANT ) {
			$class = "Limb_Chatbot\\Includes\\AI_Providers\\{$utility->get_ai_provider()::$name}\\Services\\Chatbot_Assistant_Service";
			if ( class_exists( $class ) ) {
				return new $class( $utility );
			}
			throw new Exception( Error_Codes::NOT_SUPPORTED, __( 'Assistant is not supported for current ai provider.', 'limb-chatbot' ) );
		}
		throw new Exception( Error_Codes::NOT_SUPPORTED, __( 'Unsupported chatbot.', 'limb-chatbot' ) );
	}

}