<?php

namespace Limb_Chatbot\Includes\Factories;

use Limb_Chatbot\Includes\Data_Objects\Chatbot;
use Limb_Chatbot\Includes\Exceptions\Error_Codes;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Repositories\Chat_Meta_Repository;
use Limb_Chatbot\Includes\Repositories\Chat_Repository;
use Limb_Chatbot\Includes\Services\Chat_Service;


/**
 * Factory for creating chat service instances based on chatbot type.
 *
 * @since 1.0.0
 */
class Chat_Service_Factory {

	/**
	 * Create a chat service instance for the given chatbot.
	 *
	 * @param Chatbot $chatbot The chatbot instance.
	 * @return Chat_Service|object Chat service instance.
	 * @throws Exception When chatbot type is not supported or assistant service is missing.
	 *
	 * @since 1.0.0
	 */
	public function make( Chatbot $chatbot ) {
		$chat_repository      = new Chat_Repository();
		$chat_meta_repository = new Chat_Meta_Repository();
		if ( $chatbot->get_type() === Chatbot::TYPE_ASSISTANT ) {
			$class = "Limb_Chatbot\\Includes\\AI_Providers\\{$chatbot->get_ai_provider()::$name}\\Services\\Chat_Assistant_Service";
			if ( class_exists( $class ) ) {
				return new $class( $chatbot, $chat_repository, $chat_meta_repository );
			}
			throw new Exception( Error_Codes::NOT_SUPPORTED, __( 'Assistant is missing.', 'limb-chatbot' ) );
		} elseif ( $chatbot->get_type() === Chatbot::TYPE_SIMPLE ) {
			return new Chat_Service( $chatbot, $chat_repository, $chat_meta_repository );
		}
		throw new Exception( Error_Codes::NOT_SUPPORTED, __( 'Invalid chatbot type', 'limb-chatbot' ) );
	}
}