<?php

namespace Limb_Chatbot\Includes\Data_Objects;

/**
 * Represents a content segment from a dataset source.
 *
 * @since 1.0.0
 */
class Segment {

	/**
	 * The type of the source (e.g., post, term).
	 *
	 * @var null|string
	 *
	 * @since 1.0.0
	 */
	public ?string $source_type;

	/**
	 * The sub-type of the source (e.g., page, post type).
	 *
	 * @var null|string
	 *
	 * @since 1.0.0
	 */
	public ?string $source_sub_type;

	/**
	 * The ID of the source.
	 *
	 * @var int|string|null
	 *
	 * @since 1.0.0
	 */
	public $source_id;

	/**
	 * Optional heading for the segment.
	 *
	 * @var string|null
	 *
	 * @since 1.0.0
	 */
	public ?string $heading = null;

	/**
	 * The main content of the segment.
	 *
	 * @var null|string
	 *
	 * @since 1.0.0
	 */
	public ?string $content;

	/**
	 * The order of the segment within the dataset.
	 *
	 * @var null|int
	 *
	 * @since 1.0.0
	 */
	public ?int $order = 0;

	/**
	 * Segment constructor.
	 *
	 * @param  array  $data  Array containing the segment properties:
	 *                    - source_type (string)
	 *                    - source_sub_type (string)
	 *                    - source_id (int|string)
	 *                    - heading (string|null)
	 *                    - content (string)
	 *                    - order (int)
	 *
	 * @since 1.0.0
	 *
	 */
	public function __construct( array $data ) {
		$this->source_type     = $data['source_type'];
		$this->source_sub_type = $data['source_sub_type'];
		$this->source_id       = $data['source_id'];
		$this->heading         = $data['heading'] ?? null;
		$this->content         = $data['content'];
		$this->order           = (int) ( $data['order'] ?? 0 );
	}

	/**
	 * Get a textual representation of the segment's context.
	 *
	 * @return string Context string or '(null)' if unavailable.
	 * @since 1.0.0
	 *
	 */
	public function get_context() {
		return $this->context_factory();
	}

	/**
	 * Factory to build context string depending on source type.
	 *
	 * @return string Context string or '(null)'.
	 * @since 1.0.0
	 *
	 */
	private function context_factory() {
		if ( $this->source_id && is_numeric( $this->source_id ) ) {
			if ( $this->source_type == Dataset::SOURCE_POST ) {
				return $this->get_post_context();
			} elseif ( $this->source_type == Dataset::SOURCE_TERM ) {
				return $this->get_term_context();
			}
		}

		return '(null)';
	}

	/**
	 * Build context string for a WP_Post source.
	 *
	 * @return string Context string or '(null)'.
	 * @since 1.0.0
	 *
	 */
	private function get_post_context() {
		$post = get_post( $this->source_id );
		if ( $post instanceof \WP_Post ) {
			$summary = $post->post_excerpt ?: ( wp_trim_words( wp_strip_all_tags( $post->post_content ), 30 ) . '...' );
			$context = implode( "\n", [
				"- Page title: {$post->post_title}",
				"- Page type: {$this->source_sub_type}",
				"- Page summary: $summary"
			] );
		}

		return $context ?? '(null)';
	}

	/**
	 * Build context string for a WP_Term source.
	 *
	 * @return string Currently returns '(null)', implementation pending.
	 * @since 1.0.0
	 *
	 */
	private function get_term_context() {
		return '(null)'; // TODO: implement later
	}
}
