<?php

namespace Limb_Chatbot\Includes\Data_Objects;

use Limb_Chatbot\Includes\Services\Helper;

class Parameter extends WPDB_Data_Object {

	const TABLE_NAME = 'lbaic_parameters';
	const FILLABLE = [
		'id',
		'action_id',
		'name',
		'label',
		'type',
		'required',
		'order',
		'question',
		'default_value',
		'placeholder',
		'config',
		'validation_rules',
		'suggestions'
	];

	/**
	 * Parameter type constants
	 */
	const TYPE_TEXT = 'text';
	const TYPE_TEXTAREA = 'textarea';
	const TYPE_EMAIL = 'email';
	const TYPE_URL = 'url';
	const TYPE_BUTTONS = 'buttons';
	const TYPE_VECTOR_SEARCH = 'vector_search';

	public ?int $action_id;
	public ?string $name;
	public ?string $label;
	public ?string $type;
	public ?int $required;
	public ?int $order;
	public ?string $question;
	public ?string $default_value;
	public ?string $placeholder;
	public ?array $config;
	public ?array $validation_rules;
	public ?array $suggestions;

	/**
	 * Meta properties related to this chat.
	 *
	 * @var array
	 * @since 1.0.0
	 */
	protected array $meta_properties = [ 'search_results' ];

	/**
	 * Array of included related objects.
	 * @json_excluded
	 *
	 * @var array
	 * @since 1.0.0
	 */
	public array $included = [];

	public function __construct( $instance = null ) {
		if ( is_array( $instance ) && ! empty( $instance['validation_rules'] ) && $this->isJson( $instance['validation_rules'] ) ) {
			$instance['validation_rules'] = Helper::maybe_json_decode( $instance['validation_rules'] );
		}
		if ( is_array( $instance ) && ! empty( $instance['suggestions'] ) && $this->isJson( $instance['suggestions'] ) ) {
			$instance['suggestions'] = Helper::maybe_json_decode( $instance['suggestions'] );
		}
		if ( is_array( $instance ) && ! empty( $instance['config'] ) && $this->isJson( $instance['config'] ) ) {
			$instance['config'] = Helper::maybe_json_decode( $instance['config'] );
		}
		parent::__construct( $instance );
	}

	public function get_action_id(): ?int {
		return $this->action_id;
	}

	public function set_action_id( ?int $action_id ): void {
		$this->action_id = $action_id;
	}

	public function get_name(): ?string {
		return $this->name;
	}

	public function set_name( ?string $name ): void {
		$this->name = $name;
	}

	public function get_type(): ?string {
		return $this->type;
	}

	public function set_type( ?string $type ): void {
		$this->type = $type;
	}

	public function get_required(): ?string {
		return $this->required;
	}

	public function set_required( ?string $required ): void {
		$this->required = $required;
	}

	public function format_description() {
		$parts = [];

		// Label
		if ( $this->get_label() ) {
			$parts[] = $this->get_label();
		}

		// Default value
		if ( $default = $this->get_default_value() ) {
			$parts[] = 'Default value: ' . $default;
		}

		// Placeholder (optional)
		if ( $placeholder = $this->get_placeholder() ) {
			$parts[] = 'Placeholder: ' . $placeholder;
		}

		// Validation rules
		if ( $rules = $this->get_validation_rules( true ) ) {
			$parts[] = 'Validation rules: ' . $rules;
		}

		// Suggestions
		if ( $suggestions = $this->get_suggestions() ) {
			$parts[] = '<<< { "suggestions": ' . json_encode( $suggestions, JSON_UNESCAPED_UNICODE ) . ' } >>>';
		}

		return implode( '. ', $parts ) . '.';
	}

	public function get_label(): ?string {
		return $this->label;
	}

	public function set_label( ?string $label ): void {
		$this->label = $label;
	}

	public function get_default_value(): ?string {
		return $this->default_value;
	}

	public function set_default_value( ?string $default_value ): void {
		$this->default_value = $default_value;
	}

	public function get_placeholder(): ?string {
		return $this->placeholder;
	}

	public function set_placeholder( ?string $placeholder ): void {
		$this->placeholder = $placeholder;
	}

	public function get_validation_rules( $format = false ) {
		if ( ! $format ) {
			return $this->validation_rules;
		}
		$res = '';
		foreach ( $this->validation_rules as $key => $rule ) {
			$res .= "$key: $rule,";
		}

		return rtrim( $res, ',' );
	}

	public function set_validation_rules( ?array $validation_rules ): void {
		$this->validation_rules = $validation_rules;
	}

	public function get_suggestions(): ?array {
		return $this->suggestions;
	}

	public function set_suggestions( ?array $suggestions ): void {
		$this->suggestions = $suggestions;
	}

	public function get_question(): ?string {
		return $this->question;
	}

	public function set_question( ?string $question ): void {
		$this->question = $question;
	}

	public function get_config(): ?array {
		return $this->config;
	}

	public function set_config( ?array $config ): void {
		$this->config = $config;
	}

	public function get_order(): ?int {
		return $this->order;
	}

	public function set_order( ?int $order ): void {
		$this->order = $order;
	}

	public function is_searchable() {
		return in_array( $this->get_type(), [ self::TYPE_VECTOR_SEARCH ] );
	}

	public function get_config_value( $value ) {
		return ! empty( $this->config[ $value ] ) ? $this->config[ $value ] : null;
	}

	public function inject_results( array $search_results ) {
		$this->included['search_results'] = $search_results;
	}
}