<?php

namespace Limb_Chatbot\Includes\Data_Objects;

/**
 * Class Limit
 *
 * Represents a usage limit in the chatbot system.
 *
 * @since 1.0.0
 */
class Limit extends WPDB_Data_Object {

	/**
	 * @since 1.0.0
	 */
	const TYPE_USER = 'user';

	/**
	 * @since 1.0.0
	 */
	const TYPE_GUEST = 'guest';

	/**
	 * @since 1.0.0
	 */
	const TYPE_INPUT = 'input';

	/**
	 * @since 1.0.0
	 */
	const TYPE_OUTPUT = 'output';

	/**
	 * @since 1.0.0
	 */
	const TYPE_GENERAL = 'general';

	/**
	 * Unit constant for tokens.
	 *
	 * @since 1.0.0
	 */
	const UNIT_TOKEN = 1;

	/**
	 * Unit constant for messages per chat.
	 *
	 * @since 1.0.0
	 */
	const UNIT_MESSAGE_PER_CHAT = 2;

	/**
	 * Unit constant for new chat count.
	 *
	 * @since 1.0.0
	 */
	const UNIT_NEW_CHAT = 4;

	/**
	 * Unit constant for cost.
	 *
	 * @since 1.0.0
	 */
	const UNIT_COST = 3;

	/**
	 * Status enabled.
	 *
	 * @since 1.0.0
	 */
	const STATUS_ENABLED = 1;

	/**
	 * Status disabled.
	 *
	 * @since 1.0.0
	 */
	const STATUS_DISABLED = 0;

	/**
	 * The type of the limit.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $type;

	/**
	 * The unit type of the limit.
	 *
	 * @var int|null
	 * @since 1.0.0
	 */
	public ?int $unit;

	/**
	 * The value of the limit.
	 *
	 * @var float
	 * @since 1.0.0
	 */
	public float $value;

	/**
	 * The period (in seconds or defined unit) for the limit.
	 *
	 * @var int|null
	 * @since 1.0.0
	 */
	public ?int $period;

	/**
	 * The status of the limit (enabled/disabled).
	 *
	 * @var int
	 * @since 1.0.0
	 */
	public int $status = self::STATUS_DISABLED;

	/**
	 * Associated Chatbot instance.
	 *
	 * @var Chatbot|null
	 * @since 1.0.0
	 */
	public ?Chatbot $chatbot = null;

	/**
	 * Get the type of the limit.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_type(): string {
		return $this->type;
	}

	/**
	 * Set the type of the limit.
	 *
	 * @param  string  $type
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_type( string $type ): void {
		$this->type = $type;
	}

	/**
	 * Get the value of the limit.
	 *
	 * @return float
	 * @since 1.0.0
	 */
	public function get_value(): float {
		return $this->value;
	}

	/**
	 * Set the value of the limit.
	 *
	 * @param  float  $value
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_value( float $value ): void {
		$this->value = $value;
	}

	/**
	 * Get the period of the limit.
	 *
	 * @return int|null
	 * @since 1.0.0
	 */
	public function get_period(): ?int {
		return $this->period;
	}

	/**
	 * Set the period of the limit.
	 *
	 * @param  int|null  $period
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_period( ?int $period ): void {
		$this->period = $period;
	}

	/**
	 * Get the unit type of the limit.
	 *
	 * @return int|null
	 * @since 1.0.0
	 */
	public function get_unit(): ?int {
		return $this->unit;
	}

	/**
	 * Set the unit type of the limit.
	 *
	 * @param  int|null  $unit
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_unit( ?int $unit ): void {
		$this->unit = $unit;
	}

	/**
	 * Get the status of the limit.
	 *
	 * @return int
	 * @since 1.0.0
	 */
	public function get_status(): int {
		return $this->status;
	}

	/**
	 * Set the status of the limit.
	 *
	 * @param  int  $status
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_status( int $status ): void {
		$this->status = $status;
	}

	/**
	 * Check if the limit is enabled.
	 *
	 * @return bool True if enabled, false otherwise.
	 * @since 1.0.0
	 */
	public function is_enabled(): bool {
		return $this->status === self::STATUS_ENABLED;
	}

	/**
	 * Get the associated Chatbot instance.
	 *
	 * @return Chatbot|null
	 * @since 1.0.0
	 */
	public function get_chatbot(): ?Chatbot {
		return $this->chatbot;
	}

	/**
	 * Set the associated Chatbot instance.
	 *
	 * @param  Chatbot|null  $chatbot
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_chatbot( ?Chatbot $chatbot ): void {
		$this->chatbot = $chatbot;
	}

	/**
	 * Get the usage key string for this limit based on unit.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_usage_key(): string {
		$unit_mapper = [
			self::UNIT_TOKEN            => 'tokens',
			self::UNIT_NEW_CHAT         => 'new_chat',
			self::UNIT_MESSAGE_PER_CHAT => 'messages_per_chat',
			self::UNIT_COST             => 'costs',
		];

		return 'current_' . $unit_mapper[ $this->unit ] . '_usage';
	}
}
