<?php

namespace Limb_Chatbot\Includes\Data_Objects;

/**
 * Class Lead_Field
 *
 * Represents a lead field definition with label and data type.
 * Lead fields are reusable across multiple lead captures.
 *
 * @package Limb_Chatbot\Includes\Data_Objects
 * @since 1.0.13
 */
class Lead_Field extends WPDB_Data_Object {

	/**
	 * Database table name.
	 */
	const TABLE_NAME = 'lbaic_lead_fields';

	/**
	 * Attributes allowed for mass assignment.
	 *
	 * @var array
	 */
	const FILLABLE = [
		'id',
		'field_key',
		'label',
		'data_type',
	];

	/**
	 * Data type: String
	 */
	const DATA_TYPE_TEXT = 'text';

	/**
	 * Data type: Number
	 */
	const DATA_TYPE_NUMBER = 'number';

	/**
	 * Data type: Email
	 */
	const DATA_TYPE_EMAIL = 'email';

	/**
	 * Data type: Phone
	 */
	const DATA_TYPE_PHONE = 'phone';

	/**
	 * Data type: Date
	 */
	const DATA_TYPE_DATE = 'date';

	/**
	 * Field ID
	 *
	 * @var int|null
	 */
	public ?int $id;

	/**
	 * Field key/identifier
	 *
	 * @var string
	 */
	public string $field_key;

	/**
	 * Field display label
	 *
	 * @var string
	 */
	public string $label;

	/**
	 * Data type for value storage
	 *
	 * @var string
	 */
	public string $data_type = self::DATA_TYPE_TEXT;

	/**
	 * Get field key
	 *
	 * @return string
	 * @since 1.0.13
	 */
	public function get_field_key(): string {
		return $this->field_key;
	}

	/**
	 * Set field key
	 *
	 * @param string $key
	 *
	 * @return void
	 * @since 1.0.13
	 */
	public function set_field_key( string $key ): void {
		$this->field_key = $key;
	}

	/**
	 * Get field label
	 *
	 * @return string
	 * @since 1.0.13
	 */
	public function get_label(): string {
		return $this->label;
	}

	/**
	 * Set field label
	 *
	 * @param string $label
	 *
	 * @return void
	 * @since 1.0.13
	 */
	public function set_label( string $label ): void {
		$this->label = $label;
	}

	/**
	 * Get data type
	 *
	 * @return string
	 * @since 1.0.13
	 */
	public function get_data_type(): string {
		return $this->data_type;
	}

	/**
	 * Set data type
	 *
	 * @param string $type
	 *
	 * @return void
	 * @since 1.0.13
	 */
	public function set_data_type( string $type ): void {
		$this->data_type = $type;
	}

	/**
	 * Find or create a field by key
	 *
	 * @param string $field_key Field key/identifier
	 * @param string $label Field label
	 * @param string $data_type Data type (default: string)
	 *
	 * @return Lead_Field
	 * @since 1.0.13
	 */
	public static function find_or_create( string $field_key, string $label, string $data_type = self::DATA_TYPE_TEXT ): Lead_Field {
		// Try to find existing field
		$existing = self::where( [ 'field_key' => $field_key ] )->first();
		if ( $existing instanceof Lead_Field ) {
			return $existing;
		}

		// Create new field
		$field = new self();
		$field->set_field_key( $field_key );
		$field->set_label( $label );
		$field->set_data_type( $data_type );
		$field->save();

		return $field;
	}

	/**
	 * Find field by key
	 *
	 * @param string $field_key
	 *
	 * @return Lead_Field|null
	 * @since 1.0.13
	 */
	public static function find_by_key( string $field_key ): ?Lead_Field {
		return self::where( [ 'field_key' => $field_key ] )->first();
	}

	/**
	 * Find field by ID
	 *
	 * @param int $id
	 *
	 * @return Lead_Field|null
	 * @since 1.0.13
	 */
	public static function find_by_id( int $id ): ?Lead_Field {
		return self::where( [ 'id' => $id ] )->first();
	}
}
