<?php

namespace Limb_Chatbot\Includes\Data_Objects;

use Limb_Chatbot\Includes\Services\Helper;

/**
 * Class Conversation_State
 *
 * Represents the state of a chatbot conversation.
 * Handles chat context, user preferences, intent history, and AI analysis data.
 *
 * @package Limb_Chatbot\Includes\Data_Objects
 * @since 1.0.0
 */
class Conversation_State extends WPDB_Data_Object {

	/**
	 * Database table name.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	const TABLE_NAME = 'lbaic_conversation_states';

	/**
	 * Fillable fields for mass assignment.
	 *
	 * @var array
	 * @since 1.0.0
	 */
	const FILLABLE = array(
		'chat_uuid',
		'topic',
		'entities',
		'summary',
		'confidence',
		'sentiment',
		'user_preferences',
		'user_intent_history',
	);

	/**
	 * Default confidence value if none is provided.
	 *
	 * @since 1.0.0
	 */
	public const DEFAULT_CONFIDENCE = 0.5;

	/**
	 * Maximum number of items allowed in array fields (entities, preferences, intents).
	 *
	 * @since 1.0.0
	 */
	public const MAX_ARRAY_LENGTH = 50;

	/**
	 * Minimum string similarity threshold (0-1) for detecting duplicates.
	 *
	 * @since 1.0.0
	 */
	public const STRING_SIMILARITY_THRESHOLD = 0.8;

	/**
	 * Maximum topic similarity before considering it a shift (0-1).
	 *
	 * @since 1.0.0
	 */
	public const TOPIC_SHIFT_THRESHOLD = 0.3;

	/**
	 * Unique identifier for the chat session.
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $chat_uuid = null;

	/**
	 * Current conversation topic.
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $topic = null;

	/**
	 * Entities recognized in the conversation.
	 *
	 * @var array|null
	 * @since 1.0.0
	 */
	public ?array $entities = [];

	/**
	 * Summary of the conversation so far.
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $summary = null;

	/**
	 * Confidence score of the AI's current understanding.
	 *
	 * @var float|null
	 * @since 1.0.0
	 */
	public ?float $confidence = null;

	/**
	 * Sentiment of the conversation (e.g., positive, neutral, negative).
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $sentiment = null;

	/**
	 * User preferences captured during the conversation.
	 *
	 * @var array|null
	 * @since 1.0.0
	 */
	public ?array $user_preferences = [];

	/**
	 * History of user intents throughout the conversation.
	 *
	 * @var array|null
	 * @since 1.0.0
	 */
	public ?array $user_intent_history = [];

	/**
	 * Conversation_State constructor.
	 *
	 * Initializes the object and decodes JSON fields if necessary.
	 *
	 * @param  array|null  $instance  Optional associative array of initial values.
	 *
	 * @since 1.0.0
	 */
	public function __construct( $instance = null ) {
		if ( is_array( $instance ) ) {
			if ( ! empty( $instance['entities'] ) && $this->isJson( $instance['entities'] ) ) {
				$instance['entities'] = Helper::maybe_json_decode( $instance['entities'] );
			}
			if ( ! empty( $instance['user_intent_history'] ) && $this->isJson( $instance['user_intent_history'] ) ) {
				$instance['user_intent_history'] = Helper::maybe_json_decode( $instance['user_intent_history'] );
			}
			if ( ! empty( $instance['user_preferences'] ) && $this->isJson( $instance['user_preferences'] ) ) {
				$instance['user_preferences'] = Helper::maybe_json_decode( $instance['user_preferences'] );
			}
		}

		parent::__construct( $instance );
	}

	/**
	 * Get the most recent user intent.
	 *
	 * @return string|null Latest intent or null if none exists.
	 * @since 1.0.0
	 */
	public function get_intent(): ?string {
		return ! empty( $this->user_intent_history ) ? $this->user_intent_history[ count( $this->user_intent_history ) - 1 ] : null;
	}

	/**
	 * Get the current conversation topic.
	 *
	 * @return string|null
	 * @since 1.0.0
	 */
	public function get_topic(): ?string {
		return $this->topic;
	}

	/**
	 * Set the current conversation topic.
	 *
	 * @param  string|null  $topic
	 *
	 * @since 1.0.0
	 */
	public function set_topic( ?string $topic ): void {
		$this->topic = $topic;
	}

	/**
	 * Get entities recognized in the conversation.
	 *
	 * @return array|null
	 * @since 1.0.0
	 */
	public function get_entities(): ?array {
		return $this->entities;
	}

	/**
	 * Set entities for the conversation.
	 *
	 * @param  array|null  $entities
	 *
	 * @since 1.0.0
	 */
	public function set_entities( ?array $entities ): void {
		$this->entities = $entities;
	}

	/**
	 * Get conversation summary.
	 *
	 * @return string|null
	 * @since 1.0.0
	 */
	public function get_summary(): ?string {
		return $this->summary;
	}

	/**
	 * Set conversation summary.
	 *
	 * @param  string|null  $summary
	 *
	 * @since 1.0.0
	 */
	public function set_summary( ?string $summary ): void {
		$this->summary = $summary;
	}

	/**
	 * Get AI confidence score.
	 *
	 * @return float|null
	 * @since 1.0.0
	 */
	public function get_confidence(): ?float {
		return $this->confidence;
	}

	/**
	 * Set AI confidence score.
	 *
	 * @param  float|null  $confidence
	 *
	 * @since 1.0.0
	 */
	public function set_confidence( ?float $confidence ): void {
		$this->confidence = $confidence;
	}

	/**
	 * Get chat UUID.
	 *
	 * @return string|null
	 * @since 1.0.0
	 */
	public function get_chat_uuid(): ?string {
		return $this->chat_uuid;
	}

	/**
	 * Get conversation sentiment.
	 *
	 * @return string|null
	 * @since 1.0.0
	 */
	public function get_sentiment(): ?string {
		return $this->sentiment;
	}

	/**
	 * Set conversation sentiment.
	 *
	 * @param  string|null  $sentiment
	 *
	 * @since 1.0.0
	 */
	public function set_sentiment( ?string $sentiment ): void {
		$this->sentiment = $sentiment;
	}

	/**
	 * Get user intent history.
	 *
	 * @return array|null
	 * @since 1.0.0
	 */
	public function get_user_intent_history(): ?array {
		return $this->user_intent_history;
	}

	/**
	 * Set user intent history.
	 *
	 * @param  array|null  $user_intent_history
	 *
	 * @since 1.0.0
	 */
	public function set_user_intent_history( ?array $user_intent_history ): void {
		$this->user_intent_history = $user_intent_history;
	}

	/**
	 * Get user preferences.
	 *
	 * @return array|null
	 * @since 1.0.0
	 */
	public function get_user_preferences(): ?array {
		return $this->user_preferences;
	}

	/**
	 * Set user preferences.
	 *
	 * @param  array|null  $user_preferences
	 *
	 * @since 1.0.0
	 */
	public function set_user_preferences( ?array $user_preferences ): void {
		$this->user_preferences = $user_preferences;
	}

	/**
	 * Convert object to array.
	 *
	 * @return array Associative array of object properties.
	 * @since 1.0.0
	 */
	public function to_array(): array {
		return [
			'chat_uuid'           => $this->chat_uuid,
			'topic'               => $this->topic,
			'entities'            => $this->entities,
			'summary'             => $this->summary,
			'confidence'          => $this->confidence,
			'sentiment'           => $this->sentiment,
			'user_preferences'    => $this->user_preferences,
			'user_intent_history' => $this->user_intent_history,
		];
	}

}
