<?php

namespace Limb_Chatbot\Includes\Data_Objects;

/**
 * Data object representing a chat participant.
 *
 * Provides typed properties, constants, and accessor methods
 * for managing participants of a chat (e.g., user, admin, moderator).
 *
 * @package Limb_Chatbot\Includes\Data_Objects
 * @since 1.0.0
 */
class Chat_Participant extends WPDB_Data_Object {

	/**
	 * Fillable fields for database operations.
	 *
	 * @since 1.0.0
	 */
	const FILLABLE = array( 'id', 'chat_id', 'user_id', 'role', 'joined_at' );

	/**
	 * Database table name associated with this model.
	 *
	 * @since 1.0.0
	 */
	const TABLE_NAME = 'lbaic_chat_participants';

	/**
	 * Role constant: regular user.
	 *
	 * @since 1.0.0
	 */
	const ROLE_USER = 'user';

	/**
	 * Role constant: admin user.
	 *
	 * @since 1.0.0
	 */
	const ROLE_ADMIN = 'admin';

	/**
	 * Role constant: moderator user.
	 *
	 * @since 1.0.0
	 */
	const ROLE_AGENT = 'agent';

	/**
	 * Chat ID to which the participant belongs.
	 *
	 * @var int|null
	 * @since 1.0.0
	 */
	public ?int $chat_id;

	/**
	 * User ID of the participant.
	 *
	 * @var int|null
	 * @since 1.0.0
	 */
	public ?int $user_id;

	/**
	 * Role of the participant (user, admin, moderator).
	 *
	 * @var string|null
	 * @since 1.0.0
	 */
	public ?string $role;

	/**
	 * Timestamp when the participant joined the chat.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $joined_at;

	/**
	 * Get the chat ID.
	 *
	 * @return int|null Chat ID.
	 * @since 1.0.0
	 */
	public function get_chat_id(): ?int {
		return $this->chat_id;
	}

	/**
	 * Set the chat ID.
	 *
	 * @param  int|null  $chat_id  Chat ID.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_chat_id( ?int $chat_id ): void {
		$this->chat_id = $chat_id;
	}

	/**
	 * Get the user ID.
	 *
	 * @return int|null User ID.
	 * @since 1.0.0
	 */
	public function get_user_id(): ?int {
		return $this->user_id;
	}

	/**
	 * Set the user ID.
	 *
	 * @param  int|null  $user_id  User ID.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_user_id( ?int $user_id ): void {
		$this->user_id = $user_id;
	}

	/**
	 * Get the role of the participant.
	 *
	 * @return string|null Role of the participant.
	 * @since 1.0.0
	 */
	public function get_role(): ?string {
		return $this->role;
	}

	/**
	 * Set the role of the participant.
	 *
	 * @param  string|null  $role  Role (user, admin, moderator).
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_role( ?string $role ): void {
		$this->role = $role;
	}

	/**
	 * Get the join timestamp.
	 *
	 * @return string Join timestamp.
	 * @since 1.0.0
	 */
	public function get_joined_at(): string {
		return $this->joined_at;
	}

	/**
	 * Set the join timestamp.
	 *
	 * @param  string|null  $joined_at  Timestamp when participant joined.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function set_joined_at( ?string $joined_at ): void {
		$this->joined_at = $joined_at;
	}

	public function user() {
		return Chatbot_User::find( $this->get_user_id() );
	}
}
