<?php

namespace Limb_Chatbot\Includes\Chatbot_Tools;

use Limb_Chatbot\Includes\Utilities\Chatbot_Utility;
use Limb_Chatbot\Includes\Data_Objects\Tool_Calls_Message;

/**
 * Abstract base class for all chatbot tools.
 *
 * Provides a standardized interface for defining chatbot tools, including
 * their name, description, parameter schema, and execution logic.
 *
 * @since 1.0.0
 */
abstract class Chatbot_Tool {

	/**
	 * Tool name used for registration and retrieval.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	protected string $name;

	/**
	 * Tool type (default: 'function').
	 *
	 * @var string
	 * @since 1.0.0
	 */
	protected string $type = self::TYPE_FUNCTION;

	/**
	 * Tool description explaining purpose and usage.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	protected string $description = '';

	/**
	 * Tool parameters schema for validation and documentation.
	 *
	 * @var array
	 * @since 1.0.0
	 */
	protected array $parameters_schema;

	/**
	 * Constant for function-type tools.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public const TYPE_FUNCTION = 'function';

	/**
	 * Constructor.
	 *
	 * Initializes the tool's name, description, and parameters schema
	 * by calling the corresponding abstract methods.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->parameters_schema = $this->define_parameters_schema();
		$this->name              = $this->define_name();
		$this->description       = $this->define_description();
	}

	/**
	 * Define the tool's internal name.
	 *
	 * @return string Tool name.
	 * @since 1.0.0
	 */
	abstract protected function define_name(): string;

	/**
	 * Define the tool's description and usage details.
	 *
	 * @return string Description of the tool.
	 * @since 1.0.0
	 */
	abstract protected function define_description(): string;

	/**
	 * Define the schema for the tool's parameters.
	 *
	 * Should return a JSON Schema-like array describing required parameters.
	 *
	 * @return array Parameters schema.
	 * @since 1.0.0
	 */
	abstract protected function define_parameters_schema(): array;

	/**
	 * Execute the tool's main functionality.
	 *
	 * @param array $params Tool parameters provided by AI or system.
	 * @param Tool_Calls_Message $message Original message that triggered the tool.
	 * @param Chatbot_Utility $utility Chatbot utility instance.
	 *
	 * @return array Execution result with status and optional data.
	 * @since 1.0.0
	 */
	abstract public function execute(array $params, Tool_Calls_Message $message, Chatbot_Utility $utility): array;

	/**
	 * Get the tool's name.
	 *
	 * @return string Tool name.
	 * @since 1.0.0
	 */
	public function get_name(): string {
		return $this->name;
	}

	/**
	 * Get the tool's type.
	 *
	 * @return string Tool type (e.g., 'function').
	 * @since 1.0.0
	 */
	public function get_type(): string {
		return $this->type;
	}

	/**
	 * Get the tool's description.
	 *
	 * @return string Tool description.
	 * @since 1.0.0
	 */
	public function get_description(): string {
		return $this->description;
	}

	/**
	 * Get the tool's parameter schema.
	 *
	 * @return array Parameters schema.
	 * @since 1.0.0
	 */
	public function get_parameters_schema(): array {
		return $this->parameters_schema;
	}
}
