<?php

namespace Limb_Chatbot\Includes\Api\V1\Controllers;

use Exception;
use Limb_Chatbot\Includes\Services\Data_Deletion_Service;
use Limb_Chatbot\Includes\Services\Helper;
use Limb_Chatbot\Includes\Services\Install;
use Limb_Chatbot\Includes\Services\Migration_Service;
use Limb_Chatbot\Includes\Services\Seeder_Service;
use Limb_Chatbot\Includes\Services\Session_Manager;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use WP_REST_Server;

/**
 * REST API controller for core plugin operations.
 *
 * Handles plugin-level operations following REST principles.
 * Currently supports data deletion, but can be extended for other core operations.
 *
 * @since 1.0.0
 */
class Plugin_Controller extends Rest_Controller {

	/**
	 * Route base for the controller.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	protected $rest_base = 'plugin';

	/**
	 * Registers REST routes for the controller.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function register_routes(): void {
		// DELETE /limb/chatbot/v1/plugin/data - Delete all plugin data
		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/data',
			array(
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_plugin_data' ),
					'permission_callback' => array( $this, 'permission_callback' ),
				),
			)
		);
		// POST /limb/chatbot/v1/plugin/migrate - Run the migrations
		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/migrate',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'migrate' ),
					'permission_callback' => array( $this, 'permission_callback' ),
				),
			)
		);
		// GET /limb/chatbot/v1/plugin/migration-status - Check migration status
		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/migration-status',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_migration_status' ),
					'permission_callback' => array( $this, 'permission_callback' ),
				),
			)
		);
		// GET /limb/chatbot/v1/plugin/nonce - Generate a nonce for frontend use
		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/nonce',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'generate_nonce' ),
					'permission_callback' => '__return_true',
				),
			)
		);
	}

	/**
	 * Deletes all plugin data via REST API.
	 *
	 * This endpoint deletes all plugin tables and options without checking
	 * the lbaic.plugin.remove_plugin_data setting.
	 *
	 * Follows REST principle: DELETE operation removes the resource (plugin data).
	 *
	 * @param  WP_REST_Request  $request  Incoming REST request.
	 *
	 * @return WP_REST_Response|WP_Error REST response or error.
	 * @since 1.0.0
	 */
	public function delete_plugin_data( WP_REST_Request $request ) {
		try {
			$deletion_service = Data_Deletion_Service::instance();
			if ( $deletion_service->delete_all_data() ) {
				$install = Install::instance();
				$install->create_tables();
				$install->create_roles();
				( new Seeder_Service( true ) )->seed();
			}


			return rest_ensure_response( array( 'success' => true ) );
		} catch ( Exception $e ) {
			Helper::log( $e, __METHOD__ );

			return Helper::get_wp_error( $e );
		}
	}

	/**
	 * Execute database migrations via REST API.
	 *
	 * This endpoint checks if the database needs migration and executes all
	 * necessary migrations to bring the database schema up to date with the
	 * current plugin version.
	 *
	 * Response includes:
	 * - success: Boolean indicating migration success
	 * - version_changed: Boolean indicating if version was updated
	 * - from_version: Original database version
	 * - to_version: Target plugin version
	 * - migrations_run: Array of executed migrations with details
	 * - message: Human-readable status message
	 * - errors: Array of error messages (if any)
	 *
	 * @param  WP_REST_Request  $request  Incoming REST request.
	 *
	 * @return WP_REST_Response|WP_Error REST response with migration results or error.
	 * @since 1.0.0
	 */
	public function migrate( WP_REST_Request $request ) {
		try {
			$migration_service = new Migration_Service();
			$results           = $migration_service->migrate();

			return rest_ensure_response( $results );
		} catch ( Exception $e ) {
			Helper::log( $e, __METHOD__ );

			return Helper::get_wp_error( $e );
		}
	}

	/**
	 * Get migration status without executing migrations.
	 *
	 * This endpoint checks if the database needs migration and returns
	 * information about the current state without executing any migrations.
	 *
	 * Response includes:
	 * - current_version: Current plugin version from code
	 * - db_version: Database version from options
	 * - needs_migration: Boolean indicating if migration is needed
	 * - up_to_date: Boolean indicating if database is current
	 *
	 * @param  WP_REST_Request  $request  Incoming REST request.
	 *
	 * @return WP_REST_Response|WP_Error REST response with status or error.
	 * @since 1.0.0
	 */
	public function get_migration_status( WP_REST_Request $request ) {
		try {
			$migration_service = new Migration_Service();

			return rest_ensure_response(
				array(
					'current_version' => $migration_service->get_current_version(),
					'db_version'      => $migration_service->get_db_version(),
					'needs_migration' => ! $migration_service->is_up_to_date(),
					'up_to_date'      => $migration_service->is_up_to_date(),
				)
			);
		} catch ( Exception $e ) {
			Helper::log( $e, __METHOD__ );

			return Helper::get_wp_error( $e );
		}
	}

	/**
	 * Generate a nonce for frontend use via REST API.
	 *
	 * This endpoint generates a WordPress REST API nonce that can be used
	 * by the frontend to authenticate subsequent API requests. The nonce
	 * should be included in the X-WP-Nonce header for authenticated requests.
	 *
	 * @param  WP_REST_Request  $request  Incoming REST request.
	 *
	 * @return WP_REST_Response|WP_Error REST response with migration results or error.
	 * @since 1.0.0
	 */
	public function generate_nonce( WP_REST_Request $request ) {
		try {
			$session_manager = Session_Manager::instance();
			$nonce           = $session_manager->get_nonce();
			$expires         = $session_manager->get_nonce_expiration();

			// Set cache-control headers to prevent caching of nonces
			// This ensures each request gets a fresh nonce and avoids cache issues
			add_filter( 'rest_post_dispatch', function ( $response ) {
				$response->header( 'Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0' );
				$response->header( 'Pragma', 'no-cache' );
				$response->header( 'Expires', '0' );

				return $response;
			} );

			return rest_ensure_response(
				array(
					'nonce'   => $nonce,
					'expires' => $expires,
				)
			);
		} catch ( Exception $e ) {
			Helper::log( $e, __METHOD__ );

			return Helper::get_wp_error( $e );
		}
	}

	/**
	 * Checks whether the current user has permission to manage plugin core operations.
	 *
	 * Only allows logged-in users with administrative capabilities.
	 *
	 * @param  WP_REST_Request  $request  Incoming REST request.
	 *
	 * @return bool True if authorized, false otherwise.
	 * @since 1.0.0
	 */
	public function permission_callback( $request ): bool {
		// Only allow admin users to manage core plugin operations
		return current_user_can( 'manage_options' );
	}
}


