<?php

namespace Limb_Chatbot\Includes\AI_Providers\Open_Ai\Utilities;

use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\AI_Providers\Open_Ai\Endpoints\Vector_Store_Endpoint;
use Limb_Chatbot\Includes\Exceptions\Exception;

/**
 * Utility class for managing vector store operations with the OpenAI provider.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Open_Ai\Utilities
 * @since 1.0.0
 */
class Vector_Store_Utility {

	/**
	 * Default timeout for vector store operations, in seconds.
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const DEFAULT_TIMEOUT = 60;

	/**
	 * Files utility or related object.
	 *
	 * @since 1.0.0
	 * @var mixed
	 */
	public $global_utility;

	/**
	 * Files handler or collection used by this utility.
	 *
	 * @since 1.0.0
	 * @var mixed
	 */
	public $files;

	/**
	 * Assistant instance associated with this vector store utility.
	 *
	 * @since 1.0.0
	 * @var mixed
	 */
	public $assistant;

	/**
	 * Constructor.
	 *
	 * Initializes the vector store utility with files and assistant instances.
	 *
	 * @since 1.0.0
	 *
	 * @param mixed $files     Files handler or collection.
	 * @param mixed $assistant Assistant instance.
	 */
	public function __construct( $files, $assistant ) {
		$this->files     = $files;
		$this->assistant = $assistant;
	}

	/**
	 * Retrieves the configuration object associated with the assistant.
	 *
	 * @since 1.0.0
	 *
	 * @return Config|null Configuration object or null if not found.
	 */
	public function get_config() {
		return Config::find( $this->assistant->get_config() );
	}

	/**
	 * Returns the timeout value for vector store operations.
	 *
	 * @since 1.0.0
	 *
	 * @return int Timeout in seconds.
	 */
	public function get_timeout() {
		return self::DEFAULT_TIMEOUT;
	}

	/**
	 * Creates a new vector store entry via the endpoint.
	 *
	 * @return array|null Result of the create operation.
	 * @throws Exception
	 * @since 1.0.0
	 *
	 */
	public function create() {
		return $this->get_endpoint_instance()->create();
	}

	/**
	 * Returns an instance of the vector store endpoint.
	 *
	 * @since 1.0.0
	 *
	 * @return Vector_Store_Endpoint Endpoint instance.
	 */
	public function get_endpoint_instance() {
		return new Vector_Store_Endpoint( $this );
	}

	/**
	 * Returns the files handler or collection.
	 *
	 * @since 1.0.0
	 *
	 * @return mixed Files handler or collection.
	 */
	public function get_files() {
		return $this->files;
	}

	/**
	 * Returns the assistant instance.
	 *
	 * @since 1.0.0
	 *
	 * @return mixed Assistant instance.
	 */
	public function get_assistant() {
		return $this->assistant;
	}

	/**
	 * Returns the AI provider ID from the assistant.
	 *
	 * @since 1.0.0
	 *
	 * @return mixed AI provider identifier.
	 */
	public function get_ai_provider_id() {
		return $this->assistant->get_ai_provider_id();
	}
}