<?php

namespace Limb_Chatbot\Includes\AI_Providers\Open_Ai\Data_Schemas;

use Limb_Chatbot\Includes\AI_Providers\Open_Ai\Open_Ai;
use Limb_Chatbot\Includes\Data_Objects\AI_Model;
use Limb_Chatbot\Includes\Data_Objects\Config;
use Limb_Chatbot\Includes\Scheme_Interface;
use Limb_Chatbot\Includes\Data_Objects\Setting as Setting_Data_Object;

/**
 * Class Setting
 *
 * Defines validation and default rules for OpenAI provider settings.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Open_Ai\Data_Schemas
 * @since 1.0.0
 */
class Setting implements Scheme_Interface {

	/**
	 * Returns validation rules and default values for OpenAI settings.
	 *
	 * @return array<string, array<string, mixed>> Validation rules and defaults.
	 *
	 * @since 1.0.0
	 */
	public static function rules(): array {
		$prefix = Setting_Data_Object::SETTING_PREFIX . 'ai_providers.' . Open_Ai::$id;

		return [
			$prefix . '.utilities.chatbot.ai_model_id'       => [
				'name'              => __( 'Chatbot model id', 'limb-chatbot' ),
				'type'              => 'integer',
				'default'           => ( $model = AI_Model::find_by_name( 'gpt-4o-mini' ) ) ? $model->get_id() : null,
				'validate_callback' => function ( $value ): bool {
					return (bool) AI_Model::count( [ 'id' => $value ] );
				},
				'required'          => true
			],
			$prefix . '.utilities.chatbot.config_id'       => [
				'type'              => 'integer',
				'name'              => __( 'Chatbot API Key', 'limb-chatbot' ),
				'required'          => false,
				'validate_callback' => function ( $value ): bool {
					return (bool) Config::count( [ 'id' => $value ] );
				},
			],
			$prefix . '.utilities.embedding.dimension'       => [
				'name'     => __( 'Embedding dimension', 'limb-chatbot' ),
				'type'     => 'integer',
				'default'  => 1536,
				'required' => true,
			],
			$prefix . '.utilities.embedding.encoding_format' => [
				'name'     => __( 'Embedding encoding format', 'limb-chatbot' ),
				'type'     => 'string',
				'default'  => 'float',
				'enum'     => [ 'float' ],
				'required' => true,
			],
			$prefix . '.utilities.embedding.ai_model_id'     => [
				'name'              => __( 'Embedding model id', 'limb-chatbot' ),
				'type'              => 'integer',
				'default'           => ( $model = AI_Model::find_by_name( 'text-embedding-3-small' ) ) ? $model->get_id() : null,
				'validate_callback' => function ( $value ): bool {
					return (bool) AI_Model::count( [ 'id' => $value ] );
				},
				'required'          => true
			],
			$prefix . '.utilities.embedding.config_id'       => [
				'type'              => 'integer',
				'name'              => __( 'Embedding API Key', 'limb-chatbot' ),
				'required'          => false,
				'validate_callback' => function ( $value ): bool {
					return (bool) Config::count( [ 'id' => $value ] );
				},
			],
			$prefix . '.utilities.fine-tuning.ai_model_id'   => [
				'name'              => __( 'Fine-tuning model id', 'limb-chatbot' ),
				'type'              => 'integer',
				'default'           => ( $model = AI_Model::find_by_name( 'gpt-4o-mini' ) ) ? $model->get_id() : null,
				'validate_callback' => function ( $value ): bool {
					return (bool) AI_Model::count( [ 'id' => $value ] );
				},
				'required'          => true
			],
			$prefix . '.utilities.fine-tuning.config_id'     => [
				'type'              => 'integer',
				'name'              => __( 'Fine-tuning API Key', 'limb-chatbot' ),
				'required'          => false,
				'validate_callback' => function ( $value ): bool {
					return (bool) Config::count( [ 'id' => $value ] );
				},
			],
			$prefix . '.utilities.copilot.ai_model_id'       => [
				'name'              => __( 'Copilot model id', 'limb-chatbot' ),
				'type'              => 'integer',
				'default'           => ( $model = AI_Model::find_by_name( 'gpt-4o-mini' ) ) ? $model->get_id() : null,
				'validate_callback' => function ( $value ): bool {
					return (bool) AI_Model::count( [ 'id' => $value ] );
				},
				'required'          => true
			],
			$prefix . '.utilities.copilot.config_id'       => [
				'type'              => 'integer',
				'name'              => __( 'Copilot API Key', 'limb-chatbot' ),
				'required'          => false,
				'validate_callback' => function ( $value ): bool {
					return (bool) Config::count( [ 'id' => $value ] );
				},
			],
		];
	}
}