<?php

namespace Limb_Chatbot\Includes\AI_Providers\Grok\Seeders;

use Exception;
use Limb_Chatbot\Includes\AI_Providers\Grok\Grok;
use Limb_Chatbot\Includes\Data_Objects\AI_Model;

/**
 * Class AI_Models_Seeder
 *
 * Seeder class responsible for seeding AI models specific to the xAI provider.
 * Creates or updates AI models with predefined configurations and metadata.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Grok\Seeders
 * @since 1.0.12
 */
class AI_Models_Seeder {

	/**
	 * Run the seeder to create or update AI models.
	 *
	 * Iterates over a predefined list of models, creates new entries if not present,
	 * or updates existing ones. Also updates associated metadata for each model.
	 *
	 * @return void
	 * @throws Exception If seeding fails.
	 * @since 1.0.12
	 */
	public function run() {
		$models = $this->get_models();

		foreach ( $models as $model_data ) {
			$existing = AI_Model::find_by_name( $model_data['name'] );

			if ( $existing ) {
				// Update existing model
				foreach ( $model_data as $key => $value ) {
					if ( $key !== 'metas' && property_exists( $existing, $key ) ) {
						$existing->{$key} = $value;
					}
				}
				$existing->save();
				$model = $existing;
			} else {
				// Create new model
				$model = AI_Model::create( $model_data );
			}

			// Update metas
			if ( ! empty( $model_data['metas'] ) && $model ) {
				foreach ( $model_data['metas'] as $meta_key => $meta_value ) {
					$model->update_meta( $meta_key, $meta_value );
				}
			}
		}
	}

	/**
	 * Get the list of xAI models to seed.
	 *
	 * Includes both reasoning and non-reasoning variants:
	 * - grok-4-1-fast-reasoning
	 * - grok-4-1-fast-non-reasoning
	 * - grok-4-fast-reasoning
	 * - grok-4-fast-non-reasoning
	 * - grok-3-mini
	 * - grok-3-mini-fast
	 *
	 * @return array Array of model configurations.
	 * @since 1.0.12
	 */
	private function get_models(): array {
		return array(
			// Grok 4.1 Fast (Reasoning)
			array(
				'label'              => 'Grok 4.1 Fast (Reasoning)',
				'name'               => 'grok-4-1-fast-reasoning',
				'ai_provider_id'     => Grok::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 16384,
				'context_window'     => 2000000, // 2M context
				'modalities'         => array(
					'text'  => true,
					'image' => false,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => false,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => false,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => false,
					'pdf_support'        => false,
				),
				'metas'              => array(
					'output_token_cost' => 0.0000005, // $0.50 per MTok
					'input_token_cost'  => 0.0000002, // $0.20 per MTok
				),
			),
			// Grok 4.1 Fast (Non-Reasoning)
			array(
				'label'              => 'Grok 4.1 Fast (Non Reasoning)',
				'name'               => 'grok-4-1-fast-non-reasoning',
				'ai_provider_id'     => Grok::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 16384,
				'context_window'     => 2000000, // 2M context
				'modalities'         => array(
					'text'  => true,
					'image' => false,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => false,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => false,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => false,
					'vision'             => false,
					'pdf_support'        => false,
				),
				'metas'              => array(
					'output_token_cost' => 0.0000005, // $0.50 per MTok
					'input_token_cost'  => 0.0000002, // $0.20 per MTok
				),
			),
			// Grok 4 Fast (Reasoning)
			array(
				'label'              => 'Grok 4 Fast (Reasoning)',
				'name'               => 'grok-4-fast-reasoning',
				'ai_provider_id'     => Grok::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 16384,
				'context_window'     => 2000000, // 2M context
				'modalities'         => array(
					'text'  => true,
					'image' => false,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => false,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => false,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => true,
					'vision'             => false,
					'pdf_support'        => false,
				),
				'metas'              => array(
					'output_token_cost' => 0.0000005, // $0.50 per MTok
					'input_token_cost'  => 0.0000002, // $0.20 per MTok
				),
			),
			// Grok 4 Fast (Non-Reasoning)
			array(
				'label'              => 'Grok 4 Fast (Non Reasoning)',
				'name'               => 'grok-4-fast-non-reasoning',
				'ai_provider_id'     => Grok::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 16384,
				'context_window'     => 2000000, // 2M context
				'modalities'         => array(
					'text'  => true,
					'image' => false,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => false,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => false,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => false,
					'vision'             => false,
					'pdf_support'        => false,
				),
				'metas'              => array(
					'output_token_cost' => 0.0000005, // $0.50 per MTok
					'input_token_cost'  => 0.0000002, // $0.20 per MTok
				),
			),
			// Grok 3 Mini
			array(
				'label'              => 'Grok 3 Mini',
				'name'               => 'grok-3-mini',
				'ai_provider_id'     => Grok::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 16384,
				'context_window'     => 131072, // 131K context
				'modalities'         => array(
					'text'  => true,
					'image' => false,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => false,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => false,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => false,
					'vision'             => false,
					'pdf_support'        => false,
				),
				'metas'              => array(
					'output_token_cost' => 0.0000005, // $0.50 per MTok
					'input_token_cost'  => 0.0000003, // $0.30 per MTok
				),
			),
			// Grok 3 Mini Fast
			array(
				'label'              => 'Grok 3 Mini Fast',
				'name'               => 'grok-3-mini-fast',
				'ai_provider_id'     => Grok::$id,
				'fine_tuned'         => false,
				'status'             => AI_Model::STATUS_ACTIVE,
				'output_token_limit' => 16384,
				'context_window'     => 131072, // 131K context
				'modalities'         => array(
					'text'  => true,
					'image' => false,
					'audio' => false,
				),
				'endpoints'          => array(
					'chat_completions'  => true,
					'realtime'          => false,
					'batch'             => false,
					'embeddings'        => false,
					'speech_generation' => false,
					'translation'       => false,
					'responses'         => false,
					'assistants'        => false,
					'fine_tuning'       => false,
					'image_generation'  => false,
					'transcription'     => false,
					'moderation'        => false,
				),
				'features'           => array(
					'streaming'          => true,
					'structured_outputs' => true,
					'distillation'       => false,
					'function_calling'   => true,
					'fine_tuning'        => false,
					'predicted_outputs'  => false,
					'extended_thinking'  => false,
					'vision'             => false,
					'pdf_support'        => false,
				),
				'metas'              => array(
					'output_token_cost' => 0.0000005, // $0.50 per MTok
					'input_token_cost'  => 0.0000003, // $0.30 per MTok
				),
			),
		);
	}
}
