<?php

namespace Limb_Chatbot\Includes\AI_Providers\Grok\Endpoints\Chat_Completion;

use Limb_Chatbot\Includes\Data_Objects\Tool_Calls_Message as Parent_Tool_Calls_Message;

/**
 * Class Tool_Calls_Message
 *
 * Represents a message that contains tool call instructions from xAI,
 * extending the base Tool_Calls_Message structure.
 *
 * xAI uses OpenAI-compatible format for tool calls.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Grok\Endpoints\Chat_Completion
 * @since 1.0.12
 */
class Tool_Calls_Message extends Parent_Tool_Calls_Message {

	/**
	 * Array of tool calls received from the xAI response.
	 *
	 * @var array|null
	 * @since 1.0.12
	 */
	public ?array $tool_calls_data = null;

	/**
	 * Get the tool calls array in a normalized format.
	 *
	 * Returns tool calls in OpenAI-compatible format for use with Chatbot_Tools.
	 *
	 * @return array Array of tool call objects.
	 * @since 1.0.12
	 */
	public function get_tool_calls(): array {
		if ( empty( $this->tool_calls_data ) ) {
			return array();
		}

		$tool_calls = array();
		foreach ( $this->tool_calls_data as $tool_call ) {
			if ( is_object( $tool_call ) ) {
				$tool_calls[] = $tool_call;
			} else {
				$tool_calls[] = (object) array(
					'id'       => $tool_call['id'],
					'type'     => $tool_call['type'] ?? 'function',
					'function' => (object) array(
						'name'      => $tool_call['function']['name'],
						'arguments' => $tool_call['function']['arguments'],
					),
				);
			}
		}

		return $tool_calls;
	}

	/**
	 * Get the tool calls array for API request.
	 *
	 * @return array Array of tool calls in API format.
	 * @since 1.0.12
	 */
	public function get_tool_calls_array(): array {
		if ( empty( $this->tool_calls_data ) ) {
			return array();
		}

		$tool_calls = array();
		foreach ( $this->tool_calls_data as $tool_call ) {
			if ( is_object( $tool_call ) ) {
				$tool_calls[] = array(
					'id'       => $tool_call->id,
					'type'     => $tool_call->type ?? 'function',
					'function' => array(
						'name'      => $tool_call->function->name,
						'arguments' => $tool_call->function->arguments,
					),
				);
			} else {
				$tool_calls[] = $tool_call;
			}
		}

		return $tool_calls;
	}

	/**
	 * Set the tool calls from API response.
	 *
	 * @param  array  $tool_calls  Tool calls from xAI response.
	 *
	 * @return void
	 * @since 1.0.12
	 */
	public function set_tool_calls( $tool_calls ): void {
		$this->tool_calls_data = $tool_calls;
	}

	/**
	 * Set the tool calls from array format.
	 *
	 * @param  array  $tool_calls  Tool calls in array format.
	 *
	 * @return void
	 * @since 1.0.12
	 */
	public function set_tool_calls_from_array( array $tool_calls ): void {
		$this->tool_calls_data = $tool_calls;
	}

	/**
	 * Create Tool_Calls_Message from xAI API response.
	 *
	 * @param  object  $message  The message object from API response.
	 *
	 * @return self|null The Tool_Calls_Message instance or null if no tool calls.
	 * @since 1.0.12
	 */
	public static function from_response( object $message ): ?self {
		if ( empty( $message->tool_calls ) ) {
			return null;
		}

		$instance = self::make( array(
			'role' => 'assistant',
		) );

		$instance->set_tool_calls( $message->tool_calls );

		return $instance;
	}
}
