<?php

namespace Limb_Chatbot\Includes\AI_Providers\Gemini\Utilities;

use Limb_Chatbot\Includes\AI_Providers\Gemini\Endpoints\Generate_Content_Endpoint;
use Limb_Chatbot\Includes\AI_Providers\Gemini\Gemini;
use Limb_Chatbot\Includes\Data_Objects\Message;
use Limb_Chatbot\Includes\Exceptions\Exception;
use Limb_Chatbot\Includes\Services\Collection;
use Limb_Chatbot\Includes\Utilities\Copilot_Utility as Global_Utility;
use Limb_Chatbot\Includes\Data_Objects\Config;

/**
 * Utility class for managing copilot operations using the Gemini provider.
 *
 * @package Limb_Chatbot\Includes\AI_Providers\Gemini\Utilities
 * @since 1.0.0
 */
class Copilot_Utility {

	/**
	 * Configuration object associated with this utility.
	 *
	 * May be null if no configuration is found or set.
	 *
	 * @since 1.0.0
	 * @var Config|null
	 */
	public ?Config $config;

	/**
	 * Global copilot utility instance for shared operations.
	 *
	 * @since 1.0.0
	 * @var Global_Utility
	 */
	public Global_Utility $global_utility;

	/**
	 * Timeout duration in seconds for the embedding request.
	 *
	 * @since 1.0.0
	 * @var int|null
	 */
	public ?int $timeout = 60;

	/**
	 * API version string.
	 *
	 * @var string
	 * @since 1.0.0
	 */
	public string $version = 'v1beta';

	/**
	 * Endpoint instance handling fine-tuning API calls.
	 *
	 * @since 1.0.0
	 * @var Generate_Content_Endpoint
	 */
	public Generate_Content_Endpoint $endpoint;

	/**
	 * Constructor.
	 *
	 * Initializes the utility with a global utility instance, sets the default config if missing,
	 * and initializes the fine-tuning endpoint.
	 *
	 * @param  Global_Utility  $global_utility  Global fine-tuning utility instance.
	 *
	 * @since 1.0.0
	 *
	 */
	public function __construct( Global_Utility $global_utility ) {
		$this->global_utility = $global_utility;
		if ( is_null( $this->global_utility->config ) ) {
			$this->config = Config::where( [ 'default' => true, 'related_to' => Gemini::class ] )->first();
		}
		$this->endpoint = new Generate_Content_Endpoint( $this );
	}

	/**
	 * Returns the configuration object used by this utility.
	 *
	 * @return Config|null Configuration object or null if not set.
	 * @since 1.0.0
	 *
	 */
	public function get_config(): ?Config {
		return $this->config;
	}

	/**
	 * Timeout duration in seconds for the copilot operation.
	 *
	 * @return int|null
	 * @since 1.0.0
	 */
	public function get_timeout(): ?int {
		return $this->timeout;
	}

	/**
	 * Get API version.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public function get_version(): string {
		return $this->version;
	}

	/**
	 * Proxy method to retrieve a search query from the endpoint.
	 *
	 * @param  mixed  $messages  Messages or collection of messages to process.
	 *
	 * @return Message The search query returned by the endpoint.
	 *
	 * @throws Exception
	 * @since 1.0.0
	 */
	public function is_vector_search_applicable( $messages ) {
		return $this->endpoint->is_vector_search_applicable( $messages );
	}

	/**
	 * Analyze a single conversation turn using the configured endpoint.
	 *
	 * @param  array  $messages  Array of Message objects representing the current conversation turn.
	 *
	 * @return mixed The result of the analysis from the endpoint.
	 * @throws Exception
	 * @since 1.0.0
	 */
	public function analyze_turn($messages){
		return $this->endpoint->analyze_turn($messages);
	}

	/**
	 * Generate simple text response from AI using the configured endpoint.
	 *
	 * This method is used to generate straightforward text responses based on
	 * the provided messages. It extracts just the text content from the AI's response.
	 *
	 * @param  Collection  $messages  Collection of Message objects for the AI.
	 *
	 * @return Message The generated text response.
	 * @throws Exception
	 * @since 1.0.0
	 */
	public function generate_data_collect_question( $messages, $stream = false) {
		return $this->endpoint->generate_data_collect_question( $messages, $stream );
	}

	/**
	 * Generate a beautiful success message for action completion
	 *
	 * Generates a warm, celebratory message based on the action context.
	 * Delegates to the endpoint to call the AI provider.
	 *
	 * @param  Collection  $messages  Collection of Message objects with system and user prompts
	 *
	 * @return Message The generated success message
	 * @throws Exception If message generation fails
	 * @since 1.0.0
	 */
	public function generate_action_success_message( $messages, $stream = false ) {
		return $this->endpoint->generate_action_success_message( $messages, $stream );
	}

	/**
	 * Generate action success message with extracted variables context
	 *
	 * Creates a beautiful success message that includes information from extracted variables.
	 *
	 * @param  Collection  $messages  Collection of Message objects with context and variables
	 *
	 * @return Message The generated success message with extracted variables
	 * @throws Exception If message generation fails
	 * @since 1.0.0
	 */
	public function generate_action_success_message_with_context( $messages, $stream = false ) {
		return $this->endpoint->generate_action_success_message( $messages, $stream );
	}
}